<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class Utf8SanitizationMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Sanitize request data
        $this->sanitizeRequest($request);
        
        // Get the response
        $response = $next($request);
        
        // Sanitize response data if it's JSON
        if ($response->headers->get('Content-Type') === 'application/json') {
            $this->sanitizeJsonResponse($response);
        }
        
        return $response;
    }
    
    /**
     * Sanitize request data
     */
    private function sanitizeRequest(Request $request): void
    {
        // Sanitize all input data
        $allInput = $request->all();
        $sanitizedInput = $this->recursiveSanitize($allInput);
        
        // Replace the request input with sanitized data
        $request->replace($sanitizedInput);
    }
    
    /**
     * Sanitize JSON response
     */
    private function sanitizeJsonResponse(Response $response): void
    {
        try {
            $content = $response->getContent();
            $data = json_decode($content, true);
            
            if (json_last_error() === JSON_ERROR_NONE) {
                $sanitizedData = $this->recursiveSanitize($data);
                $response->setContent(json_encode($sanitizedData));
            }
        } catch (\Exception $e) {
            // If sanitization fails, log and continue
            \Log::warning('UTF-8 sanitization failed for response', [
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Recursively sanitize data
     */
    private function recursiveSanitize($data)
    {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->recursiveSanitize($value);
            }
            return $data;
        }
        
        if (is_string($data)) {
            return $this->sanitizeString($data);
        }
        
        return $data;
    }
    
    /**
     * Sanitize individual string
     */
    private function sanitizeString(string $string): string
    {
        if (empty($string)) {
            return $string;
        }
        
        // Ensure the string is valid UTF-8
        if (!mb_check_encoding($string, 'UTF-8')) {
            $string = mb_convert_encoding($string, 'UTF-8', 'UTF-8');
        }
        
        // Remove invalid UTF-8 sequences (but preserve valid Unicode characters)
        $string = mb_convert_encoding($string, 'UTF-8', 'UTF-8');
        
        // Remove only dangerous control characters (preserve Unicode characters)
        // Allow: \x09 (tab), \x0A (LF), \x0D (CR)
        // Remove: \x00-\x08, \x0B, \x0C, \x0E-\x1F, \x7F (dangerous control chars)
        $string = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $string);
        
        // DO NOT convert Unicode to ASCII - preserve Kurdish/Arabic/Unicode characters
        // The previous iconv conversion was stripping Unicode characters
        // We only want to remove invalid sequences, not convert valid Unicode
        
        return $string;
    }
}
