<?php

namespace App\Http\Controllers\pages;

use App\Http\Controllers\Controller;
use App\Models\Patient;
use App\Models\Invoice;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class HomePage extends Controller
{
    public function index()
    {
        // Get the authenticated user
        $user = Auth::user();

        // Initialize variables
        $totalPatients = 0;
        $totalInvoices = 0;
        $totalProducts = 0;
        $recentInvoices = [];
        $invoicesToday = 0;
        $invoicesThisMonth = 0;

        // Check user roles and prepare role-specific data
        if ($user->hasRole('super-admin') || $user->hasRole('supervisor')) {
            // Admin/Supervisor dashboard
            $totalPatients = Patient::count();
            $totalInvoices = Invoice::count();
            $totalProducts = Product::count();
            
            // Get today's invoices
            $invoicesToday = Invoice::whereDate('invoice_date', today())->count();
            
            // Get this month's invoices
            $invoicesThisMonth = Invoice::whereMonth('invoice_date', now()->month)
                ->whereYear('invoice_date', now()->year)
                ->count();
            
            // Get recent invoices
            $recentInvoices = Invoice::with(['patient'])
                ->latest()
                ->take(10)
                ->get();
            
            // Get invoices by status
            $invoicesByStatus = Invoice::select('status', DB::raw('count(*) as total'))
                ->groupBy('status')
                ->get();
            
            // Revenue calculations
            $totalRevenue = Invoice::where('status', 'completed')->sum('total_amount');
            $todayRevenue = Invoice::where('status', 'completed')
                ->whereDate('invoice_date', today())
                ->sum('total_amount');
            $monthRevenue = Invoice::where('status', 'completed')
                ->whereMonth('invoice_date', now()->month)
                ->whereYear('invoice_date', now()->year)
                ->sum('total_amount');
            $lastMonthRevenue = Invoice::where('status', 'completed')
                ->whereMonth('invoice_date', now()->subMonth()->month)
                ->whereYear('invoice_date', now()->subMonth()->year)
                ->sum('total_amount');
            
            // Monthly revenue for last 6 months (for chart)
            $monthlyRevenue = [];
            for ($i = 5; $i >= 0; $i--) {
                $date = now()->subMonths($i);
                $revenue = Invoice::where('status', 'completed')
                    ->whereMonth('invoice_date', $date->month)
                    ->whereYear('invoice_date', $date->year)
                    ->sum('total_amount');
                $monthlyRevenue[] = [
                    'month' => $date->format('M Y'),
                    'revenue' => $revenue
                ];
            }
            
            // Daily revenue for last 7 days (for chart)
            $dailyRevenue = [];
            for ($i = 6; $i >= 0; $i--) {
                $date = now()->subDays($i);
                $revenue = Invoice::where('status', 'completed')
                    ->whereDate('invoice_date', $date)
                    ->sum('total_amount');
                $dailyRevenue[] = [
                    'day' => $date->format('M d'),
                    'revenue' => $revenue
                ];
            }
            
            // Monthly invoice count for last 6 months
            $monthlyInvoices = [];
            for ($i = 5; $i >= 0; $i--) {
                $date = now()->subMonths($i);
                $count = Invoice::whereMonth('invoice_date', $date->month)
                    ->whereYear('invoice_date', $date->year)
                    ->count();
                $monthlyInvoices[] = [
                    'month' => $date->format('M Y'),
                    'count' => $count
                ];
            }
            
            // Top products by sales
            $topProducts = Product::with('invoiceItems')
                ->get()
                ->map(function($product) {
                    $product->total_sold = $product->invoiceItems->sum('quantity');
                    return $product;
                })
                ->sortByDesc('total_sold')
                ->take(5)
                ->values();
            
            // Top patients by invoice count
            $topPatients = Patient::withCount('invoices')
                ->orderBy('invoices_count', 'desc')
                ->take(5)
                ->get();
            
            // New patients this month
            $newPatientsThisMonth = Patient::whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->count();
            
            // Low stock products
            $lowStockProducts = Product::where('stock_quantity', '<', 10)
                ->where('stock_quantity', '>', 0)
                ->count();
            $outOfStockProducts = Product::where('stock_quantity', '<=', 0)->count();
            
            // Revenue growth percentage
            $revenueGrowth = 0;
            if ($lastMonthRevenue > 0) {
                $revenueGrowth = (($monthRevenue - $lastMonthRevenue) / $lastMonthRevenue) * 100;
            }
            
            return view('content.pages.pages-home', compact(
                'totalPatients',
                'totalInvoices',
                'totalProducts',
                'invoicesToday',
                'invoicesThisMonth',
                'recentInvoices',
                'invoicesByStatus',
                'totalRevenue',
                'todayRevenue',
                'monthRevenue',
                'lastMonthRevenue',
                'monthlyRevenue',
                'dailyRevenue',
                'monthlyInvoices',
                'topProducts',
                'topPatients',
                'newPatientsThisMonth',
                'lowStockProducts',
                'outOfStockProducts',
                'revenueGrowth'
            ));
        } 
        elseif ($user->hasRole('cashier')) {
            // Cashier dashboard - show their invoices
            $invoicesToday = Invoice::where('user_id', $user->id)
                ->whereDate('invoice_date', today())
                ->count();
            
            $totalInvoices = Invoice::where('user_id', $user->id)->count();
            
            $recentInvoices = Invoice::where('user_id', $user->id)
                ->with(['patient'])
                ->latest()
                ->take(10)
                ->get();
            
            return view('content.pages.pages-home', compact(
                'invoicesToday',
                'totalInvoices',
                'recentInvoices'
            ));
        }
        elseif ($user->hasRole('customer')) {
            // Customer role - if you want to keep this for future patient portal
            return view('content.pages.pages-home', [
                'totalPatients' => 0,
                'totalInvoices' => 0
            ]);
        }

        // Default return
        return view('content.pages.pages-home', [
            'totalPatients' => 0,
            'totalInvoices' => 0,
            'totalProducts' => 0,
            'invoicesToday' => 0,
            'invoicesThisMonth' => 0,
            'recentInvoices' => [],
            'invoicesByStatus' => []
        ]);
    }
}
