<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use Illuminate\View\View;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\TransactionStoreRequest;
use App\Http\Requests\TransactionUpdateRequest;

class TransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Transaction::class);

        $search = $request->get('search', '');

        $transactions = Transaction::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view(
            'app.transactions.index',
            compact('transactions', 'search')
        );
    }

    public function showCurrencyForm()
    {
        return view('app.transactions.currency-transaction-form'); // The Blade view that includes the Livewire component
    }
    public function showForm()
    {
        return view('app.transactions.transaction-form'); // The Blade view that includes the Livewire component
    }

    public function showFormTest()
    {
        return view('app.transactions.transaction-form-test'); // The Blade view that includes the Livewire component
    }
    public function approval()
    {
        return view('app.account_transactions.approval'); // The Blade view that includes the Livewire component
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Transaction::class);

        $currencies = Currency::pluck('currency_name', 'id');

        return view(
            'app.transactions.create',
            compact('currencies', 'currencies')
        );
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TransactionStoreRequest $request): RedirectResponse
    {
        $this->authorize('create', Transaction::class);

        $validated = $request->validated();

        $transaction = Transaction::create($validated);

        return redirect()
            ->route('transactions.edit', $transaction)
            ->withSuccess(__('crud.common.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Transaction $transaction): View
    {
        $this->authorize('view', $transaction);

        return view('app.transactions.show', compact('transaction'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Transaction $transaction): View
    {
        $this->authorize('update', $transaction);

        $currencies = Currency::pluck('currency_name', 'id');

        return view(
            'app.transactions.edit',
            compact('transaction', 'currencies', 'currencies')
        );
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        TransactionUpdateRequest $request,
        Transaction $transaction
    ): RedirectResponse {
        $this->authorize('update', $transaction);

        $validated = $request->validated();

        $transaction->update($validated);

        return redirect()
            ->route('transactions.edit', $transaction)
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        Transaction $transaction
    ): RedirectResponse {
        $this->authorize('delete', $transaction);

        try {
            // Check for related account transactions
            if ($transaction->accountTransactions()->count() > 0) {
                return redirect()
                    ->route('transactions.index')
                    ->withErrors(['error' => 'Cannot delete transaction. This transaction has associated account transactions.']);
            }

            $transaction->delete();

            return redirect()
                ->route('transactions.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('transactions.index')
                    ->withErrors(['error' => 'Cannot delete transaction. This transaction has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('transactions.index')
                ->withErrors(['error' => 'An error occurred while deleting the transaction. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('transactions.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
