<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\AccountTransaction;
use App\Models\Notification;
use OneSignal;
use App\Services\TransactionNotificationService;
use Illuminate\Support\Facades\Auth;

class TestNotificationController extends Controller
{
    protected $notificationService;

    public function __construct(TransactionNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Show the test notification page
     */
    public function index()
    {
        // Get all users for testing (since we use user_id as external_id)
        $usersWithPlayerIds = User::with('customer')->get();

        // Get recent transactions for testing
        $recentTransactions = AccountTransaction::with(['from_account.customer.user'])
            ->latest()
            ->limit(10)
            ->get();

        return view('test-notifications', compact('usersWithPlayerIds', 'recentTransactions'));
    }

    /**
     * Send test notification to specific user using user_id as external_id
     */
    public function sendTestNotification(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'message' => 'nullable|string|max:255',
        ]);

        try {
            $user = User::findOrFail($request->user_id);
            
            $message = isset($request->message) ? $request->message : 'Test notification from LA DERMA - ' . now()->format('Y-m-d H:i:s');
            
            // Use OneSignal facade directly (same as pakezh-web)
            $externalUserId = (string) $user->id;
            
            // Store notification in database
            $notification = Notification::create([
                'type' => 'test',
                'title' => 'تست نوێکاری',
                'body' => $message,
                'user_id' => $user->id,
                'data' => [
                    'type' => 'test',
                    'timestamp' => now()->toISOString(),
                    'sent_by' => isset(Auth::user()->name) ? Auth::user()->name : 'System',
                ],
            ]);

            $response = OneSignal::sendNotificationCustom([
                'include_external_user_ids' => [$externalUserId],
                'contents' => ["en" => $message],
                'data' => [
                    'type' => 'test',
                    'timestamp' => now()->toISOString(),
                    'sent_by' => isset(Auth::user()->name) ? Auth::user()->name : 'System',
                ],
                'app_id' => config('onesignal.app_id')
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Test notification sent and stored successfully!',
                'user_name' => $user->name,
                'user_id' => $user->id,
                'external_id' => $externalUserId,
                'notification_id' => $notification->id,
                'onesignal_response' => $response
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Send test transaction notification using existing transaction
     */
    public function sendTestTransactionNotification(Request $request)
    {
        $request->validate([
            'transaction_id' => 'required|exists:account_transactions,id',
            'notification_type' => 'required|in:transaction_completed,transaction_approved,transaction_rejected',
        ]);

        try {
            $transaction = AccountTransaction::with(['from_account.customer.user'])->findOrFail($request->transaction_id);
            
            $result = $this->notificationService->sendTransactionCreatedNotification($transaction);

            if ($result) {
                return response()->json([
                    'success' => true,
                    'message' => 'Transaction notification sent successfully!',
                    'transaction_id' => $transaction->id,
                    'transaction_amount' => $transaction->transaction_amount,
                    'transaction_type' => $transaction->transaction_type,
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to send transaction notification',
                ], 500);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Send test notification to all users with player IDs
     */
    public function sendTestToAllUsers(Request $request)
    {
        $request->validate([
            'message' => 'nullable|string|max:255',
        ]);

        try {
            $users = User::all(); // Get all users since we're using user_id as external_id
            
            if ($users->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No users found',
                ], 400);
            }

            $externalUserIds = $users->pluck('id')->map(function($id) {
                return (string) $id; // Convert to string as required by OneSignal
            })->toArray();
            
            $message = isset($request->message) ? $request->message : 'Broadcast test notification from LA DERMA - ' . now()->format('Y-m-d H:i:s');
            
            // Store notifications in database for each user
            $storedNotifications = [];
            foreach ($users as $user) {
                $notification = Notification::create([
                    'type' => 'broadcast_test',
                    'title' => 'نوێکاری گشتی',
                    'body' => $message,
                    'user_id' => $user->id,
                    'data' => [
                        'type' => 'broadcast_test',
                        'timestamp' => now()->toISOString(),
                        'sent_by' => isset(Auth::user()->name) ? Auth::user()->name : 'System',
                    ],
                ]);
                $storedNotifications[] = $notification->id;
            }
            
            // Use OneSignal facade directly (same as pakezh-web)
            $response = OneSignal::sendNotificationCustom([
                'include_external_user_ids' => $externalUserIds,
                'contents' => ["en" => $message],
                'data' => [
                    'type' => 'broadcast_test',
                    'timestamp' => now()->toISOString(),
                    'sent_by' => isset(Auth::user()->name) ? Auth::user()->name : 'System',
                ],
                'app_id' => config('onesignal.app_id')
            ]);
            
            return response()->json([
                'success' => true,
                'message' => "Test notification sent and stored for {$users->count()} users!",
                'user_count' => $users->count(),
                'external_ids' => $externalUserIds,
                'stored_notification_ids' => $storedNotifications,
                'onesignal_response' => $response
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get OneSignal configuration status
     */
    public function getConfigStatus()
    {
        $config = [
            'app_id' => config('onesignal.app_id') ? 'Set' : 'Not Set',
            'rest_api_key' => config('onesignal.rest_api_key') ? 'Set' : 'Not Set',
            'user_auth_key' => config('onesignal.user_auth_key') ? 'Set' : 'Not Set',
        ];

        $totalUsers = User::count();

        return response()->json([
            'config' => $config,
            'total_users' => $totalUsers,
            'notification_method' => 'external_user_id',
            'note' => 'All users can receive notifications using their user_id as external_id',
        ]);
    }
}
