<?php

namespace App\Http\Controllers;

use App\Models\Subcategory;
use App\Models\Category;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;

class SubcategoryController extends Controller
{
    public function index(Request $request): View
    {
        $this->authorize('view-any', Subcategory::class);
        $search = $request->get('search', '');
        $subcategories = Subcategory::with('category')->search($search)->latest()->paginate(50)->withQueryString();
        return view('app.subcategories.index', compact('subcategories', 'search'));
    }

    public function create(Request $request): View
    {
        $this->authorize('create', Subcategory::class);
        $categories = Category::whereIn('type', ['service', 'other'])->pluck('name', 'id');
        return view('app.subcategories.create', compact('categories'));
    }

    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Subcategory::class);
        $validated = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'name' => 'required|string|max:255',
            'cost' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'sort_order' => 'nullable|integer',
        ]);
        Subcategory::create($validated);
        return redirect()->route('subcategories.index')->withSuccess(__('crud.common.created'));
    }

    public function show(Request $request, Subcategory $subcategory): View
    {
        $this->authorize('view', $subcategory);
        $subcategory->load('category');
        return view('app.subcategories.show', compact('subcategory'));
    }

    public function edit(Request $request, Subcategory $subcategory): View
    {
        $this->authorize('update', $subcategory);
        $categories = Category::whereIn('type', ['service', 'other'])->pluck('name', 'id');
        return view('app.subcategories.edit', compact('subcategory', 'categories'));
    }

    public function update(Request $request, Subcategory $subcategory): RedirectResponse
    {
        $this->authorize('update', $subcategory);
        $validated = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'name' => 'required|string|max:255',
            'cost' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'sort_order' => 'nullable|integer',
        ]);
        $subcategory->update($validated);
        return redirect()->route('subcategories.index')->withSuccess(__('crud.common.saved'));
    }

    public function destroy(Request $request, Subcategory $subcategory): RedirectResponse
    {
        $this->authorize('delete', $subcategory);
        
        try {
            // Check for related invoice items
            if (method_exists($subcategory, 'invoiceItems') && $subcategory->invoiceItems()->count() > 0) {
                return redirect()
                    ->route('subcategories.index')
                    ->withErrors(['error' => 'Cannot delete subcategory. This subcategory has been used in invoices.']);
            }

            $subcategory->delete();
            return redirect()->route('subcategories.index')->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('subcategories.index')
                    ->withErrors(['error' => 'Cannot delete subcategory. This subcategory has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('subcategories.index')
                ->withErrors(['error' => 'An error occurred while deleting the subcategory. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('subcategories.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}

