<?php

namespace App\Http\Controllers;

use App\Models\PatientHistory;
use App\Models\Patient;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;

class PatientHistoryController extends Controller
{
    public function index(Request $request): View
    {
        $this->authorize('view-any', PatientHistory::class);
        $search = $request->get('search', '');
        $patientHistories = PatientHistory::with(['patient', 'invoice', 'createdBy'])
            ->search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();
        return view('app.patient-histories.index', compact('patientHistories', 'search'));
    }

    public function create(Request $request): View
    {
        $this->authorize('create', PatientHistory::class);
        $patients = Patient::all()->mapWithKeys(function ($patient) {
            return [$patient->id => $patient->name . ' (' . $patient->code . ')'];
        });
        $invoices = \App\Models\Invoice::pluck('invoice_number', 'id');
        return view('app.patient-histories.create', compact('patients', 'invoices'));
    }

    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', PatientHistory::class);
        $validated = $request->validate([
            'patient_id' => 'required|exists:patients,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'history_notes' => 'required|string',
            'history_date' => 'required|date',
            'history_type' => 'nullable|string|max:255',
        ]);
        $validated['created_by'] = Auth::id();
        PatientHistory::create($validated);
        return redirect()->route('patient-histories.index')->withSuccess(__('crud.common.created'));
    }

    public function show(Request $request, PatientHistory $patientHistory): View
    {
        $this->authorize('view', $patientHistory);
        $patientHistory->load(['patient', 'invoice', 'createdBy']);
        return view('app.patient-histories.show', compact('patientHistory'));
    }

    public function edit(Request $request, PatientHistory $patientHistory): View
    {
        $this->authorize('update', $patientHistory);
        $patients = Patient::all()->mapWithKeys(function ($patient) {
            return [$patient->id => $patient->name . ' (' . $patient->code . ')'];
        });
        $invoices = \App\Models\Invoice::pluck('invoice_number', 'id');
        return view('app.patient-histories.edit', compact('patientHistory', 'patients', 'invoices'));
    }

    public function update(Request $request, PatientHistory $patientHistory): RedirectResponse
    {
        $this->authorize('update', $patientHistory);
        $validated = $request->validate([
            'patient_id' => 'required|exists:patients,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'history_notes' => 'required|string',
            'history_date' => 'required|date',
            'history_type' => 'nullable|string|max:255',
        ]);
        $patientHistory->update($validated);
        return redirect()->route('patient-histories.index')->withSuccess(__('crud.common.saved'));
    }

    public function destroy(Request $request, PatientHistory $patientHistory): RedirectResponse
    {
        $this->authorize('delete', $patientHistory);
        
        try {
            $patientHistory->delete();
            return redirect()->route('patient-histories.index')->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('patient-histories.index')
                    ->withErrors(['error' => 'Cannot delete patient history. This record has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('patient-histories.index')
                ->withErrors(['error' => 'An error occurred while deleting the patient history. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('patient-histories.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
