<?php

namespace App\Http\Controllers;

use App\Models\File;
use App\Models\AccountTransaction;
use App\Models\Account;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class FileUploadController extends Controller
{
    /**
     * Show the file upload form.
     */
  

    public function showUploadForm($model, $id)
    {
        // Dynamically get the model class name
        $modelClass = "App\\Models\\" . ucfirst($model);
        $modelInstance = $modelClass::with('files')->find($id); // Eager load files

        // Check if the model instance was found
        if (!$modelInstance) {
            return redirect()->back()->withErrors(['error' => 'Model not found.']);
        }

        return view('app.accounts.file', [
            'model' => $model,
            'id' => $id,
            'files' => $modelInstance->files, // Pass files to the view
        ]);
    }


    /**
     * Handle file upload.
     */
    
     public function uploadFile(Request $request, $model, $id)
     {
         // Validate the request (file is optional), allow only image files
         $request->validate([
             'file' => 'nullable|image|mimes:jpeg,png,jpg|max:2240', // Only allow image types, max 2.2MB
         ]);
     
         // Handle the uploaded file
         if ($request->hasFile('file')) {
             // Store the file and get the path
             $filePath = $request->file('file')->store('files', 'public');
     
             // Dynamically get the model class name
             $modelClass = "App\\Models\\" . ucfirst($model);
             $modelInstance = $modelClass::find($id); // Use the ID to find the model
     
             // Check if the model instance was found
             if (!$modelInstance) {
                 return back()->withErrors(['error' => 'Model not found.']);
             }
             // Save the file record into the database directly using the File model
             \App\Models\File::create([
                 'file_path'     => $filePath,
                 'fileable_id'   => $modelInstance->id, // Use the model instance's ID
                 'fileable_type' => $modelClass,        // Use the full class name of the model
             ]);

             return redirect()->route('account-transactions.create')->with('success', 'Image uploaded successfully.');
            }
     
         return back()->with('info', 'No file uploaded.');
     }

     public function destroy($fileId)
    {
        try {
            // Find the file by ID
            $file = File::find($fileId);

            // Check if the file exists
            if (!$file) {
                return redirect()->back()->withErrors(['error' => 'File not found.']);
            }

            // Delete the file from storage
            if ($file->file_path && Storage::disk('public')->exists($file->file_path)) {
                Storage::disk('public')->delete($file->file_path);
            }

            // Delete the file record from the database
            $file->delete();

            return redirect()->back()->with('success', 'File deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'An error occurred while deleting the file. Please try again.']);
        }
    }
     
}
