<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\CustomerStoreRequest;
use App\Http\Requests\CustomerUpdateRequest;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Customer::class);

        $search = $request->get('search', '');

        $customers = Customer::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view('app.customers.index', compact('customers', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Customer::class);

        return view('app.customers.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CustomerStoreRequest $request): RedirectResponse
    {
        $this->authorize('create', Customer::class);

        try {
            $validated = $request->validated();

            $customer = Customer::create($validated);

            return redirect()
                ->route('customers.edit', $customer)
                ->withSuccess(__('crud.common.created'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while creating the customer. Please try again.']);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Customer $customer): View
    {
        $this->authorize('view', $customer);

        return view('app.customers.show', compact('customer'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Customer $customer): View
    {
        $this->authorize('update', $customer);

        return view('app.customers.edit', compact('customer'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        CustomerUpdateRequest $request,
        Customer $customer
    ): RedirectResponse {
        $this->authorize('update', $customer);

        try {
            $validated = $request->validated();

            $customer->update($validated);

            return redirect()
                ->route('customers.edit', $customer)
                ->withSuccess(__('crud.common.saved'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->withInput()
                ->withErrors(['error' => 'An error occurred while updating the customer. Please try again.']);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        Customer $customer
    ): RedirectResponse {
        $this->authorize('delete', $customer);

        try {
            // Check for related accounts
            if ($customer->accounts()->count() > 0) {
                return redirect()
                    ->route('customers.index')
                    ->withErrors(['error' => 'Cannot delete customer. This customer has associated accounts.']);
            }

            $customer->delete();

            return redirect()
                ->route('customers.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('customers.index')
                    ->withErrors(['error' => 'Cannot delete customer. This customer has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('customers.index')
                ->withErrors(['error' => 'An error occurred while deleting the customer. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('customers.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
