<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\CurrencyStoreRequest;
use App\Http\Requests\CurrencyUpdateRequest;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Session;

class CurrencyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Currency::class);

        $search = $request->get('search', '');

        $currencies = Currency::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view('app.currencies.index', compact('currencies', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Currency::class);

        return view('app.currencies.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CurrencyStoreRequest $request): RedirectResponse
    {
        $this->authorize('create', Currency::class);
    
        $validated = $request->validated();
        $validated['rate'] = 0;

        // Handle the image upload and store the image path in the 'code' field
        if ($request->hasFile('code')) {
            $imagePath = $request->file('code')->store('currencies', 'public');
            $validated['code'] = $imagePath; // Save the image path in 'code'
        }
    
        $currency = Currency::create($validated);
    
        Session::flash('message', [
            'message' => 'Currency Created Successfully',
            'type' => 'alert-success',
        ]);
    
        return redirect()
            ->route('currencies.index', $currency)
            ->withSuccess(__('crud.common.created'));
    }
    
    /**
     * Display the specified resource.
     */
    public function show(Request $request, Currency $currency): View
    {
        $this->authorize('view', $currency);

        return view('app.currencies.show', compact('currency'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Currency $currency): View
    {
        $this->authorize('update', $currency);

        return view('app.currencies.edit', compact('currency'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(CurrencyUpdateRequest $request, Currency $currency): RedirectResponse
    {
        $this->authorize('update', $currency);
    
        $validated = $request->validated();
    
        // Handle image upload if a new image is uploaded
        if ($request->hasFile('code')) {
            // Delete the old image if it exists
            if ($currency->code) {
                Storage::disk('public')->delete($currency->code);
            }
            $imagePath = $request->file('code')->store('currencies', 'public');
            $validated['code'] = $imagePath; // Save the image path in 'code'
        }
    
        $currency->update($validated);
    
        Session::flash('message', [
            'message' => 'Currency Updated Successfully',
            'type' => 'alert-success',
        ]);
    
        return redirect()
            ->route('currencies.index', $currency)
            ->withSuccess(__('crud.common.saved'));
    }
    
        
    

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        Currency $currency
    ): RedirectResponse {
        $this->authorize('delete', $currency);

        try {
            // Check for related accounts
            $hasAccounts = \App\Models\Account::where('currency_id', $currency->id)->exists();
            
            if ($hasAccounts) {
                return redirect()
                    ->route('currencies.index')
                    ->withErrors(['error' => 'Cannot delete currency. This currency is used in accounts.']);
            }

            // Check for related transactions
            $hasTransactions = \App\Models\Transaction::where('from_currency', $currency->id)
                ->orWhere('to_currency', $currency->id)
                ->exists();

            if ($hasTransactions) {
                return redirect()
                    ->route('currencies.index')
                    ->withErrors(['error' => 'Cannot delete currency. This currency is used in transactions.']);
            }

            // Delete associated image if exists
            if ($currency->code && \Illuminate\Support\Facades\Storage::disk('public')->exists($currency->code)) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($currency->code);
            }

            $currency->delete();

            return redirect()
                ->route('currencies.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('currencies.index')
                    ->withErrors(['error' => 'Cannot delete currency. This currency has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('currencies.index')
                ->withErrors(['error' => 'An error occurred while deleting the currency. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('currencies.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
