<?php

namespace App\Http\Controllers;

use App\Models\Category;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;

class CategoryController extends Controller
{
    public function index(Request $request): View
    {
        $this->authorize('view-any', Category::class);
        $search = $request->get('search', '');
        $categories = Category::search($search)->latest()->paginate(50)->withQueryString();
        return view('app.categories.index', compact('categories', 'search'));
    }

    public function create(Request $request): View
    {
        $this->authorize('create', Category::class);
        return view('app.categories.create');
    }

    public function store(Request $request): RedirectResponse
    {
        $this->authorize('create', Category::class);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:service,drug,other',
            'description' => 'nullable|string',
            'sort_order' => 'nullable|integer',
        ]);
        Category::create($validated);
        return redirect()->route('categories.index')->withSuccess(__('crud.common.created'));
    }

    public function show(Request $request, Category $category): View
    {
        $this->authorize('view', $category);
        $category->load('subcategories');
        return view('app.categories.show', compact('category'));
    }

    public function edit(Request $request, Category $category): View
    {
        $this->authorize('update', $category);
        return view('app.categories.edit', compact('category'));
    }

    public function update(Request $request, Category $category): RedirectResponse
    {
        $this->authorize('update', $category);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:service,drug,other',
            'description' => 'nullable|string',
            'sort_order' => 'nullable|integer',
        ]);
        $category->update($validated);
        return redirect()->route('categories.index')->withSuccess(__('crud.common.saved'));
    }

    public function destroy(Request $request, Category $category): RedirectResponse
    {
        $this->authorize('delete', $category);
        
        try {
            // Check for related subcategories
            if ($category->subcategories()->count() > 0) {
                return redirect()
                    ->route('categories.index')
                    ->withErrors(['error' => 'Cannot delete category. This category has associated subcategories.']);
            }

            $category->delete();
            return redirect()->route('categories.index')->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('categories.index')
                    ->withErrors(['error' => 'Cannot delete category. This category has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('categories.index')
                ->withErrors(['error' => 'An error occurred while deleting the category. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('categories.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}

