<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Cashier;
use Illuminate\View\View;
use App\Models\CashierShift;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\CashierShiftStoreRequest;
use App\Http\Requests\CashierShiftUpdateRequest;

class CashierShiftController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', CashierShift::class);

        $search = $request->get('search', '');

        $cashierShifts = CashierShift::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view(
            'app.cashier_shifts.index',
            compact('cashierShifts', 'search')
        );
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', CashierShift::class);

        $cashiers = Cashier::pluck('cashier_name', 'id');
        $users = User::pluck('name', 'id');

        return view('app.cashier_shifts.create', compact('cashiers', 'users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CashierShiftStoreRequest $request): RedirectResponse
    {
        $this->authorize('create', CashierShift::class);

        $validated = $request->validated();

        $cashierShift = CashierShift::create($validated);

        return redirect()
            ->route('cashier-shifts.edit', $cashierShift)
            ->withSuccess(__('crud.common.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, CashierShift $cashierShift): View
    {
        $this->authorize('view', $cashierShift);

        return view('app.cashier_shifts.show', compact('cashierShift'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, CashierShift $cashierShift): View
    {
        $this->authorize('update', $cashierShift);

        $cashiers = Cashier::pluck('cashier_name', 'id');
        $users = User::pluck('name', 'id');

        return view(
            'app.cashier_shifts.edit',
            compact('cashierShift', 'cashiers', 'users')
        );
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        CashierShiftUpdateRequest $request,
        CashierShift $cashierShift
    ): RedirectResponse {
        $this->authorize('update', $cashierShift);

        $validated = $request->validated();

        $cashierShift->update($validated);

        return redirect()
            ->route('cashier-shifts.edit', $cashierShift)
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        CashierShift $cashierShift
    ): RedirectResponse {
        $this->authorize('delete', $cashierShift);

        try {
            // Check for related transactions
            if ($cashierShift->transactions()->count() > 0) {
                return redirect()
                    ->route('cashier-shifts.index')
                    ->withErrors(['error' => 'Cannot delete cashier shift. This shift has associated transactions.']);
            }

            $cashierShift->delete();

            return redirect()
                ->route('cashier-shifts.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('cashier-shifts.index')
                    ->withErrors(['error' => 'Cannot delete cashier shift. This shift has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('cashier-shifts.index')
                ->withErrors(['error' => 'An error occurred while deleting the cashier shift. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('cashier-shifts.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
