<?php

namespace App\Http\Controllers;

use App\Models\Cashier;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\Account;
use App\Models\User;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\CashierStoreRequest;
use App\Http\Requests\CashierUpdateRequest;
use Illuminate\Support\Facades\Session;

class CashierController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Cashier::class);

        $search = $request->get('search', '');

        $cashiers = Cashier::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view('app.cashiers.index', compact('cashiers', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Cashier::class);

        return view('app.cashiers.create');
    }

    /**
     * Store a newly created resource in storage.
     */

     public function store(CashierStoreRequest $request)
     {
        // Authorize the action
        $this->authorize('create', Cashier::class);

        // Validate the incoming request
        $validated = $request->validated();
        
        // Check if a cashier with the same name already exists
        $cashierExists = User::where('name', $validated['cashier_name'])->exists();

        if ($cashierExists) {
            // Flash error message and redirect back
            Session::flash('message', [
                'message' => 'کاشێرێک یان بەکارهێنەرێک بەم ناوە پێشتر بوونی هەیە!',
                'type' => 'alert-danger',
            ]);

            return redirect()->back();
        }

        // Generate a user email from cashier's name
        $cashierName = $validated['cashier_name']; // Assuming 'cashier_name' is a part of the validated request
        $userEmail = strtolower(str_replace(' ', '', $cashierName)) . '@LA DERMA.com';

        // Create the user and associate with the cashier
        $user = User::create([
            'name' => $cashierName,
            'email' => $userEmail,
            'password' => bcrypt('password123'), // Generate a default password or something secure
        ]);
        
         // Assign role based on cashier type
        $cashierType = $validated['cashier_type']; // Ensure 'cashier_type' is part of the request
        if ($cashierType === 'supervisor') {
            $user->syncRoles('supervisor'); // Assign supervisor role
        } elseif ($cashierType === 'cashier') {
            $user->syncRoles('cashier'); // Assign cashier role
        }

        // Create a customer for the newly created user
        $customer = Customer::create([
            'customer_name' => 'Cashiers', // You may change this to whatever is appropriate
            'user_id' => $user->id, // Associate the user with the customer
            'phone_number' => '000000', 
            'address' => 'Cashier', 
        ]);
        // Get all currencies from the currencies table
        $currencies = Currency::all();

        // Loop through each currency and create an account for the cashier
        foreach ($currencies as $currency) {
            // Set the account title 
            $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                              ->orderBy('number', 'desc')
                              ->first();

            $newAccountTitle = $lastAccount ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) : 'CN00001';

            // Create the account for the cashier for each currency
            Account::create([
                'account_title' => $newAccountTitle,
                'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
                'balance' => 0, // Default balance, assuming starting with 0
                'customer_id' => $customer->id, // Use the ID of the "Cashiers" customer
                'currency_id' => $currency->id, // Set the currency ID
                'account_type' => 'checking',
            ]);
        }

        // Create the cashier record and assign the user_id to the cashier
        $cashier = Cashier::create(array_merge($validated, [
            'user_id' => $user->id, // Assign the user_id to the cashier
            'start_time' => now(),   // Set start_time to current time
            'end_time' => now(),     // Set end_time to current time
        ]));

        // Flash success message
        Session::flash('message', [
            'message' => 'کاشێر بەسەرکەوتووی دروستکرا',
            'type' => 'alert-success',
        ]);

        // Return the cashier resource
        return redirect()
            ->route('cashiers.index', $cashier)
            ->withSuccess(__('crud.common.created'));
    }  

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Cashier $cashier): View
    {
        $this->authorize('view', $cashier);

        return view('app.cashiers.show', compact('cashier'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Cashier $cashier): View
    {
        $this->authorize('update', $cashier);

        return view('app.cashiers.edit', compact('cashier'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        CashierUpdateRequest $request,
        Cashier $cashier
    ): RedirectResponse {
        $this->authorize('update', $cashier);

        $validated = $request->validated();

        $cashier->update($validated);
        Session::flash('message', [
            'message' => 'Cashier Updated Successfully',
            'type' => 'alert-success',
        ]);
        return redirect()
            ->route('cashiers.index', $cashier)
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        Cashier $cashier
    ): RedirectResponse {
        $this->authorize('delete', $cashier);

        try {
            // Check for related cashier shifts
            if ($cashier->cashierShifts()->count() > 0) {
                return redirect()
                    ->route('cashiers.index')
                    ->withErrors(['error' => 'Cannot delete cashier. This cashier has associated shifts.']);
            }

            $cashier->delete();
            Session::flash('message', [
                'message' => 'Cashier Deleted Successfully',
                'type' => 'alert-success',
            ]);
            return redirect()
                ->route('cashiers.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('cashiers.index')
                    ->withErrors(['error' => 'Cannot delete cashier. This cashier has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('cashiers.index')
                ->withErrors(['error' => 'An error occurred while deleting the cashier. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('cashiers.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
