<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use ZipArchive;

class BackupController extends Controller
{
    public function index()
    {
        $disk = Storage::disk('local');
        $backupPath = 'backups';
        
        // Create backups directory if it doesn't exist
        if (!$disk->exists($backupPath)) {
            $disk->makeDirectory($backupPath);
        }
        
        $files = $disk->files($backupPath);
        
        $backups = [];
        foreach ($files as $file) {
            if (substr($file, -4) == '.zip' || substr($file, -4) == '.sql') {
                $backups[] = [
                    'file_name' => basename($file),
                    'file_size' => $disk->size($file),
                    'created_at' => Carbon::parse($disk->lastModified($file)),
                    'download_link' => route('backup.download', ['filename' => basename($file)])
                ];
            }
        }
        
        // Sort backups by creation date (newest first)
        usort($backups, function ($a, $b) {
            return $b['created_at']->timestamp - $a['created_at']->timestamp;
        });
        
        return view('app.backup.index', compact('backups'));
    }
    
    public function create()
    {
        try {
            // Create a backup filename with timestamp
            $timestamp = Carbon::now()->format('Y-m-d-H-i-s');
            $filename = "backup-{$timestamp}.sql";
            $backupPath = storage_path("app/backups/{$filename}");
            
            // Ensure the backups directory exists
            if (!file_exists(dirname($backupPath))) {
                mkdir(dirname($backupPath), 0755, true);
            }
            
            // Open the SQL file for writing
            $file = fopen($backupPath, 'w');
            
            // Add SQL header with timestamp
            fwrite($file, "-- Database backup created on " . date('Y-m-d H:i:s') . "\n");
            fwrite($file, "-- ------------------------------------------------------------\n\n");
            
            // Get all tables
            $tables = DB::select('SHOW TABLES');
            
            foreach ($tables as $table) {
                $tableName = get_object_vars($table)[key(get_object_vars($table))];
                
                // Add table structure
                fwrite($file, "-- Table structure for table `{$tableName}`\n");
                fwrite($file, "DROP TABLE IF EXISTS `{$tableName}`;\n");
                
                // Get create table statement
                $createTableSql = DB::select("SHOW CREATE TABLE `{$tableName}`");
                $createTableStatement = get_object_vars($createTableSql[0])['Create Table'] ?? null;
                
                if ($createTableStatement) {
                    fwrite($file, $createTableStatement . ";\n\n");
                }
                
                // Get table data
                $rows = DB::table($tableName)->get();
                
                if (count($rows) > 0) {
                    fwrite($file, "-- Dumping data for table `{$tableName}`\n");
                    
                    // Get column names
                    $columns = Schema::getColumnListing($tableName);
                    
                    // Create INSERT statements in batches
                    $batchSize = 100;
                    $totalRows = count($rows);
                    
                    for ($i = 0; $i < $totalRows; $i += $batchSize) {
                        $batchRows = array_slice($rows->toArray(), $i, $batchSize);
                        
                        if (empty($batchRows)) {
                            continue;
                        }
                        
                        fwrite($file, "INSERT INTO `{$tableName}` (`" . implode("`, `", $columns) . "`) VALUES\n");
                        
                        $valueStrings = [];
                        foreach ($batchRows as $row) {
                            $values = [];
                            $rowArray = (array) $row;
                            
                            foreach ($columns as $column) {
                                $value = $rowArray[$column] ?? null;
                                
                                if (is_null($value)) {
                                    $values[] = "NULL";
                                } elseif (is_numeric($value)) {
                                    $values[] = $value;
                                } elseif ($column === 'properties') {
                                    // Ensure JSON is properly encoded
                                    $json = json_encode(json_decode($value, true), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
                                    $values[] = "'" . str_replace("'", "''", $json) . "'";
                                } else {
                                    $values[] = "'" . str_replace("'", "''", $value) . "'";
                                }
                            }
                            
                            $valueStrings[] = "(" . implode(", ", $values) . ")";
                        }
                        
                        fwrite($file, implode(",\n", $valueStrings) . ";\n\n");
                    }
                }
            }
            
            // Close the file
            fclose($file);
            
            // Compress the SQL file to ZIP
            $zipFilename = "backup-{$timestamp}.zip";
            $zipPath = storage_path("app/backups/{$zipFilename}");
            
            $zip = new ZipArchive();
            if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
                $zip->addFile($backupPath, basename($backupPath));
                $zip->close();
                
                // Remove the SQL file if ZIP was created successfully
                if (file_exists($zipPath)) {
                    unlink($backupPath);
                }
                
                return redirect()->route('backup.index')->with('success', 'Database backup created successfully');
            }
            
            // If ZIP creation fails, keep the SQL file
            return redirect()->route('backup.index')->with('success', 'Database backup created successfully (uncompressed)');
            
        } catch (\Exception $e) {
            Log::error('Backup exception: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            return redirect()->route('backup.index')->with('error', 'Backup failed: ' . $e->getMessage());
        }
    }
    
    public function showImportForm()
    {
        return view('app.backup.import');
    }
    
    public function import(Request $request)
    {
        // Validate the uploaded file
        $request->validate([
            'backup_file' => 'required|file|mimes:sql,zip|max:51200', // 50MB max
        ]);
        
        try {
            $file = $request->file('backup_file');
            $originalName = $file->getClientOriginalName();
            $extension = $file->getClientOriginalExtension();
            
            // Store the uploaded file temporarily
            $tempPath = $file->store('temp', 'local');
            $fullTempPath = storage_path("app/{$tempPath}");
            
            $sqlFilePath = '';
            $shouldDeleteSqlFile = false;
            
            // Handle ZIP files
            if ($extension === 'zip') {
                $zip = new ZipArchive();
                if ($zip->open($fullTempPath) === TRUE) {
                    // Extract to temporary directory
                    $extractPath = storage_path('app/temp/extracted');
                    if (!file_exists($extractPath)) {
                        mkdir($extractPath, 0755, true);
                    }
                    
                    $zip->extractTo($extractPath);
                    $zip->close();
                    
                    // Find the SQL file in the extracted contents
                    $extractedFiles = glob($extractPath . '/*.sql');
                    if (empty($extractedFiles)) {
                        throw new \Exception('No SQL file found in the ZIP archive.');
                    }
                    
                    $sqlFilePath = $extractedFiles[0];
                    $shouldDeleteSqlFile = true;
                } else {
                    throw new \Exception('Unable to open ZIP file.');
                }
            } else {
                // Direct SQL file
                $sqlFilePath = $fullTempPath;
            }
            
            // Read and execute the SQL file
            $sqlContent = file_get_contents($sqlFilePath);
            
            if (empty($sqlContent)) {
                throw new \Exception('The SQL file is empty or could not be read.');
            }
            
            // Disable foreign key checks temporarily
            DB::statement('SET FOREIGN_KEY_CHECKS=0');
            
            // Split SQL content into individual statements
            $statements = $this->splitSqlStatements($sqlContent);
            
            foreach ($statements as $statement) {
                $statement = trim($statement);
                if (!empty($statement) && !$this->isComment($statement)) {
                    DB::statement($statement);
                }
            }
            
            // Re-enable foreign key checks
            DB::statement('SET FOREIGN_KEY_CHECKS=1');
            
            // Clean up temporary files
            Storage::disk('local')->delete($tempPath);
            if ($shouldDeleteSqlFile && file_exists($sqlFilePath)) {
                unlink($sqlFilePath);
            }
            if (file_exists(dirname($sqlFilePath)) && $shouldDeleteSqlFile) {
                rmdir(dirname($sqlFilePath));
            }
            
            return redirect()->route('backup.index')->with('success', 'Database imported successfully from: ' . $originalName);
            
        } catch (\Exception $e) {
            Log::error('Import exception: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            
            // Clean up temporary files on error
            if (isset($tempPath)) {
                Storage::disk('local')->delete($tempPath);
            }
            if (isset($sqlFilePath) && $shouldDeleteSqlFile && file_exists($sqlFilePath)) {
                unlink($sqlFilePath);
                if (file_exists(dirname($sqlFilePath))) {
                    rmdir(dirname($sqlFilePath));
                }
            }
            
            return redirect()->back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }
    
    private function splitSqlStatements($sql)
    {
        // Remove comments and split by semicolons
        $sql = preg_replace('/^--.*$/m', '', $sql); // Remove single-line comments
        $sql = preg_replace('/\/\*.*?\*\//s', '', $sql); // Remove multi-line comments
        
        // Split by semicolons, but be careful with semicolons inside strings
        $statements = [];
        $current = '';
        $inString = false;
        $stringChar = '';
        
        for ($i = 0; $i < strlen($sql); $i++) {
            $char = $sql[$i];
            $prev = $i > 0 ? $sql[$i - 1] : '';
            
            if (!$inString && ($char === '"' || $char === "'")) {
                $inString = true;
                $stringChar = $char;
            } elseif ($inString && $char === $stringChar && $prev !== '\\') {
                $inString = false;
                $stringChar = '';
            }
            
            if (!$inString && $char === ';') {
                $statements[] = $current;
                $current = '';
            } else {
                $current .= $char;
            }
        }
        
        if (!empty(trim($current))) {
            $statements[] = $current;
        }
        
        return $statements;
    }
    
    private function isComment($statement)
    {
        $statement = trim($statement);
        return strpos($statement, '--') === 0 || strpos($statement, '/*') === 0;
    }
    
    public function download($filename)
    {
        $file = storage_path("app/backups/{$filename}");
        
        if (file_exists($file)) {
            return response()->download($file);
        }
        
        return redirect()->route('backup.index')->with('error', 'Backup file not found');
    }
    
    public function delete($filename)
    {
        $file = storage_path("app/backups/{$filename}");
        
        if (file_exists($file)) {
            unlink($file);
            return redirect()->route('backup.index')->with('success', 'Backup deleted successfully');
        }
        
        return redirect()->route('backup.index')->with('error', 'Backup file not found');
    }
}