<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class CustomerController extends Controller
{
    /**
     * Get customer profile
     */
    public function profile(Request $request)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        // Get accounts with balances
        $accounts = $customer->accounts()->with('currency:id,currency_name')->get();
        $accounts->each(function ($account) {
            $account->current_balance = $account->calculateTotalBalance();
        });

        return response()->json([
            'success' => true,
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'roles' => $user->getRoleNames(),
                ],
                'customer' => [
                    'id' => $customer->id,
                    'customer_name' => $customer->customer_name,
                    'phone_number' => $customer->phone_number,
                    'second_phone' => $customer->second_phone,
                    'address' => $customer->address,
                    'note' => $customer->note,
                ],
                'accounts' => $accounts,
                'total_balance' => $accounts->sum('current_balance'),
            ]
        ]);
    }

    /**
     * Update customer profile
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'nullable|string|max:255',
            'phone_number' => 'nullable|string|max:20',
            'second_phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'note' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Update user name if provided
        if ($request->has('name') && $request->name) {
            $user->update(['name' => $request->name]);
        }

        // Update customer profile
        $customer->update($request->only([
            'phone_number',
            'second_phone',
            'address',
            'note'
        ]));

        // Update customer name if name is provided
        if ($request->has('name') && $request->name) {
            $customer->update(['customer_name' => $request->name]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                ],
                'customer' => [
                    'id' => $customer->id,
                    'customer_name' => $customer->customer_name,
                    'phone_number' => $customer->phone_number,
                    'second_phone' => $customer->second_phone,
                    'address' => $customer->address,
                    'note' => $customer->note,
                ],
            ]
        ]);
    }

    /**
     * Get customer account statement
     */
    public function accountStatement(Request $request)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        // Get request parameters
        $accountId = $request->get('account_id');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');

        // If specific account and date range are provided, get detailed statement
        if ($accountId && $startDate && $endDate) {
            return $this->getDetailedAccountStatement($customer, $accountId, $startDate, $endDate);
        }

        // Default: Get accounts with balances and recent transactions
        $accounts = $customer->accounts()->with([
            'currency:id,currency_name',
            'transactions' => function ($query) {
                $query->orderBy('transaction_date', 'desc')->limit(5);
            }
        ])->get();

        // Calculate balances for each account
        $accounts->each(function ($account) {
            $account->current_balance = $account->calculateTotalBalance();
        });

        // Get recent transactions across all accounts (last 10)
        $accountIds = $accounts->pluck('id')->toArray();
        $recentTransactions = \App\Models\AccountTransaction::whereIn('from_account_id', $accountIds)
            ->orWhereIn('to_account_id', $accountIds)
            ->with(['from_account.currency', 'to_account.currency'])
            ->orderBy('transaction_date', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'customer_info' => [
                    'id' => $customer->id,
                    'customer_name' => $customer->customer_name,
                    'phone_number' => $customer->phone_number,
                    'user_id' => $user->id,
                    'user_name' => $user->name,
                    'user_email' => $user->email,
                ],
                'accounts' => $accounts->map(function ($account) {
                    return [
                        'id' => $account->id,
                        'currency_name' => $account->currency->currency_name,
                        'current_balance' => $account->current_balance,
                        'recent_transactions' => $account->transactions->map(function ($transaction) {
                            return [
                                'id' => $transaction->id,
                                'transaction_type' => $transaction->transaction_type,
                                'transaction_amount' => $transaction->transaction_amount,
                                'transaction_date' => $transaction->transaction_date,
                                'description' => $transaction->description,
                            ];
                        })
                    ];
                }),
                'recent_transactions' => $recentTransactions->map(function ($transaction) {
                    return [
                        'id' => $transaction->id,
                        'transaction_type' => $transaction->transaction_type,
                        'transaction_amount' => $transaction->transaction_amount,
                        'transaction_date' => $transaction->transaction_date,
                        'description' => $transaction->description,
                        'from_account' => $transaction->from_account ? [
                            'currency_name' => $transaction->from_account->currency->currency_name,
                        ] : null,
                        'to_account' => $transaction->to_account ? [
                            'currency_name' => $transaction->to_account->currency->currency_name,
                        ] : null,
                    ];
                }),
                'total_balance' => $accounts->sum('current_balance'),
                'statement_date' => now()->format('Y-m-d H:i:s'),
            ]
        ]);
    }

    /**
     * Get detailed account statement for specific account and date range
     */
    private function getDetailedAccountStatement($customer, $accountId, $startDate, $endDate)
    {
        // Find the specific account
        $account = $customer->accounts()->with('currency:id,currency_name')->find($accountId);
        
        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found or does not belong to customer'
            ], 404);
        }

        // Parse dates
        $start = \Carbon\Carbon::parse($startDate)->startOfDay();
        $end = \Carbon\Carbon::parse($endDate)->endOfDay();

        // Get transactions for the specific account and date range
        $transactions = \App\Models\AccountTransaction::where(function ($query) use ($accountId) {
            $query->where('from_account_id', $accountId)
                  ->orWhere('to_account_id', $accountId);
        })
        ->whereBetween('transaction_date', [$start, $end])
        ->with(['from_account.currency', 'to_account.currency'])
        ->orderBy('transaction_date', 'desc')
        ->get();

        // Calculate summary statistics
        $totalDeposits = $transactions->where('to_account_id', $accountId)->sum('transaction_amount');
        $totalWithdrawals = $transactions->where('from_account_id', $accountId)->sum('transaction_amount');
        $currentBalance = $account->calculateTotalBalance();

        // Format transactions for response
        $formattedTransactions = $transactions->map(function ($transaction) use ($accountId) {
            $isDeposit = $transaction->to_account_id == $accountId;
            $amount = $transaction->transaction_amount;
            
            return [
                'id' => $transaction->id,
                'transaction_type' => $transaction->transaction_type,
                'transaction_amount' => $amount,
                'transaction_date' => $transaction->transaction_date,
                'description' => $transaction->description,
                'is_deposit' => $isDeposit,
                'from_account' => $transaction->from_account ? [
                    'id' => $transaction->from_account->id,
                    'currency_name' => $transaction->from_account->currency->currency_name,
                ] : null,
                'to_account' => $transaction->to_account ? [
                    'id' => $transaction->to_account->id,
                    'currency_name' => $transaction->to_account->currency->currency_name,
                ] : null,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => [
                'account_info' => [
                    'id' => $account->id,
                    'currency_name' => $account->currency->currency_name,
                    'current_balance' => $currentBalance,
                ],
                'statement_period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'duration_days' => $start->diffInDays($end) + 1,
                ],
                'transactions' => $formattedTransactions,
                'summary' => [
                    'total_transactions' => $transactions->count(),
                    'total_deposits' => $totalDeposits,
                    'total_withdrawals' => $totalWithdrawals,
                    'current_balance' => $currentBalance,
                ],
                'statement_date' => now()->format('Y-m-d H:i:s'),
            ]
        ]);
    }
}