<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class CurrencyController extends Controller
{
    /**
     * Get currency exchange rates
     */
    public function rates()
    {
        // Cache currencies for 30 seconds to reduce database load for high traffic
        $currencies = Cache::remember('public_currencies_api', 30, function () {
            $currencies = Currency::whereIn('currency_name', [
                'USD', 'EURO', 'POUND', 'TURKIYE', 'TUMAN', 'EMIRATES', 'JORDAN', 'SWEED', 'NARWEJ', 'SWESRA', 'SAUDI', 'AUSTRALA', 'CANADA'
            ])
            ->select('currency_name', 'buy_rate', 'sell_rate', 'sort_order')
            ->get();
            
            // Sort currencies in the exact order specified by client
            $orderMap = [
                'USD' => 1, 'EURO' => 2, 'POUND' => 3, 'TURKIYE' => 4, 'TUMAN' => 5,
                'EMIRATES' => 6, 'JORDAN' => 7, 'SWEED' => 8, 'NARWEJ' => 9, 'SWESRA' => 10,
                'SAUDI' => 11, 'AUSTRALA' => 12, 'CANADA' => 13
            ];
            
            $currencies = $currencies->sortBy(function ($currency) use ($orderMap) {
                return $orderMap[$currency->currency_name] ?? 999;
            });
            
            return $currencies;
        });

        // Transform the data to match the expected API format
        $rates = $currencies->map(function ($currency) {
            return [
                'currency' => $currency->currency_name,
                'name' => $this->getCurrencyDisplayName($currency->currency_name),
                'flag' => $this->getCurrencyFlag($currency->currency_name),
                'buy_rate' => (float) $currency->buy_rate,
                'sell_rate' => (float) $currency->sell_rate,
            ];
        })->values()->toArray();

        return response()->json([
            'success' => true,
            'message' => 'Currency rates retrieved successfully',
            'data' => [
                'rates' => $rates,
                'location' => 'Hawler (Erbil), Iraq',
                'last_updated' => now()->toISOString(),
                'contact' => [
                    'phone' => '0751 825 3732 <br>
                    0751 466 8873 <br>
                    0751 531 4173',
                    'website' => 'www.LA DERMA.com',
                    'hours' => 'Daily 9:00 AM - 6:00 PM',
                ],
            ],
        ]);
    }

    /**
     * Get display name for currency
     */
    private function getCurrencyDisplayName($currencyName)
    {
        $displayNames = [
            'USD' => 'US Dollar',
            'EURO' => 'Euro',
            'POUND' => 'British Pound',
            'TURKIYE' => 'Turkish Lira',
            'TUMAN' => 'Iranian Toman',
            'EMIRATES' => 'UAE Dirham',
            'JORDAN' => 'Jordanian Dinar',
            'SWEED' => 'Swedish Krona',
            'NARWEJ' => 'Norwegian Krone',
            'SWESRA' => 'Swiss Franc',
            'SAUDI' => 'Saudi Riyal',
            'AUSTRALA' => 'Australian Dollar',
            'CANADA' => 'Canadian Dollar',
        ];

        return $displayNames[$currencyName] ?? $currencyName;
    }

    /**
     * Get flag emoji for currency
     */
    private function getCurrencyFlag($currencyName)
    {
        $flags = [
            'USD' => '🇺🇸',
            'EURO' => '🇪🇺',
            'POUND' => '🇬🇧',
            'TURKIYE' => '🇹🇷',
            'TUMAN' => '🇮🇷',
            'EMIRATES' => '🇦🇪',
            'JORDAN' => '🇯🇴',
            'SWEED' => '🇸🇪',
            'NARWEJ' => '🇳🇴',
            'SWESRA' => '🇨🇭',
            'SAUDI' => '🇸🇦',
            'AUSTRALA' => '🇦🇺',
            'CANADA' => '🇨🇦',
        ];

        return $flags[$currencyName] ?? '💰';
    }
}