<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AppVersion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class AppVersionController extends Controller
{
    /**
     * Get the latest app version information
     *
     * @return JsonResponse
     */
    public function getLatestVersion(): JsonResponse
    {
        try {
            // Get the latest version for the current platform
            $platform = request()->header('X-Platform', 'android'); // Default to android
            $currentVersion = request()->header('X-App-Version', '1.0.0');
            
            // Get latest version from database
            $latestVersion = AppVersion::where('platform', $platform)
                ->where('is_active', true)
                ->orderBy('version_code', 'desc')
                ->first();
            
            if (!$latestVersion) {
                return response()->json([
                    'success' => false,
                    'message' => 'No version information available'
                ], 404);
            }
            
            // Check if update is required
            $isUpdateRequired = $this->isUpdateRequired($currentVersion, $latestVersion->version);
            $isForceUpdate = $this->isForceUpdateRequired($currentVersion, $latestVersion->version);
            
            return response()->json([
                'success' => true,
                'data' => [
                    'version' => $latestVersion->version,
                    'version_code' => $latestVersion->version_code,
                    'platform' => $latestVersion->platform,
                    'is_update_required' => $isUpdateRequired,
                    'is_force_update' => $isForceUpdate,
                    'release_notes' => $latestVersion->release_notes,
                    'download_url' => $latestVersion->download_url,
                    'min_supported_version' => $latestVersion->min_supported_version,
                    'release_date' => $latestVersion->created_at->toISOString(),
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get version information',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Check if update is required
     *
     * @param string $currentVersion
     * @param string $latestVersion
     * @return bool
     */
    private function isUpdateRequired(string $currentVersion, string $latestVersion): bool
    {
        return version_compare($latestVersion, $currentVersion, '>');
    }
    
    /**
     * Check if force update is required
     *
     * @param string $currentVersion
     * @param string $latestVersion
     * @return bool
     */
    private function isForceUpdateRequired(string $currentVersion, string $latestVersion): bool
    {
        // Get the minimum supported version
        $minSupportedVersion = AppVersion::where('platform', request()->header('X-Platform', 'android'))
            ->where('is_active', true)
            ->orderBy('version_code', 'desc')
            ->first()?->min_supported_version ?? '1.0.0';
            
        // If current version is below minimum supported, force update
        return version_compare($currentVersion, $minSupportedVersion, '<');
    }
    
    /**
     * Create or update app version (Admin endpoint)
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function createOrUpdate(Request $request): JsonResponse
    {
        $request->validate([
            'version' => 'required|string',
            'version_code' => 'required|integer',
            'platform' => 'required|in:android,ios',
            'release_notes' => 'nullable|string',
            'download_url' => 'nullable|url',
            'min_supported_version' => 'nullable|string',
            'is_active' => 'boolean',
            'is_force_update' => 'boolean',
        ]);
        
        try {
            $appVersion = AppVersion::updateOrCreate(
                [
                    'platform' => $request->platform,
                    'version_code' => $request->version_code,
                ],
                [
                    'version' => $request->version,
                    'release_notes' => $request->release_notes,
                    'download_url' => $request->download_url,
                    'min_supported_version' => $request->min_supported_version ?? $request->version,
                    'is_active' => $request->boolean('is_active', true),
                    'is_force_update' => $request->boolean('is_force_update', false),
                ]
            );
            
            return response()->json([
                'success' => true,
                'message' => 'App version updated successfully',
                'data' => $appVersion
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update app version',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get all app versions (Admin endpoint)
     *
     * @return JsonResponse
     */
    public function getAllVersions(): JsonResponse
    {
        try {
            $versions = AppVersion::orderBy('platform')
                ->orderBy('version_code', 'desc')
                ->get();
                
            return response()->json([
                'success' => true,
                'data' => $versions
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get app versions',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
