<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AccountController extends Controller
{
    /**
     * Get user's accounts grouped by currency with total balance
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        $accounts = $customer->accounts()->with('currency:id,currency_name,code')->get();

        // Group accounts by currency and calculate total balance for each currency
        $currencyBalances = $accounts->groupBy('currency_id')->map(function ($currencyAccounts) {
            $currency = $currencyAccounts->first()->currency;
            $totalBalance = $currencyAccounts->sum(function ($account) {
                return $account->calculateTotalBalance();
            });

            // Debug: Log the currency data
            \Log::info('Currency Debug', [
                'currency_id' => $currency->id,
                'currency_name' => $currency->currency_name,
                'currency_code_raw' => $currency->code,
                'currency_code_final' => $currency->code ?? 'USD'
            ]);

            return [
                'id' => $currency->id,
                'currency_name' => $currency->currency_name,
                'currency_code' => $currency->code ?? 'USD',
                'total_balance' => $totalBalance,
                'account_count' => $currencyAccounts->count()
            ];
        })->values();

        return response()->json([
            'success' => true,
            'data' => $currencyBalances
        ]);
    }

    /**
     * Get account details with balance
     */
    public function show($id)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        $account = $customer->accounts()
            ->with('currency:id,currency_name')
            ->find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found'
            ], 404);
        }

        // Calculate balances
        $account->current_balance = $account->calculateTotalBalance();
        $account->approved_balance = $account->calculateBalance();
        $account->pending_balance = $account->calculateBalancePending();

        // Get transaction statistics
        $account->transaction_stats = [
            'total_transactions' => $account->getCountTransaction(),
            'total_deposits' => $account->getTotalDeposit(),
            'total_withdrawals' => $account->getTotalWithdrawal(),
            'net_balance' => $account->netBalance(),
        ];

        return response()->json([
            'success' => true,
            'data' => $account
        ]);
    }

    /**
     * Get account balance
     */
    public function balance($id)
    {
        $user = Auth::user();
        $customer = $user->customer;

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Customer profile not found'
            ], 404);
        }

        $account = $customer->accounts()->find($id);

        if (!$account) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'account_id' => $account->id,
                'account_title' => $account->account_title,
                'current_balance' => $account->calculateTotalBalance(),
                'approved_balance' => $account->calculateBalance(),
                'pending_balance' => $account->calculateBalancePending(),
                'currency' => $account->currency->currency_name ?? 'Unknown',
            ]
        ]);
    }

    /**
     * Get available currencies
     */
    public function currencies()
    {
        $currencies = Currency::select('id', 'currency_name', 'currency_code')->get();

        return response()->json([
            'success' => true,
            'data' => $currencies
        ]);
    }
}