<?php

namespace App\Http\Controllers;

use App\Models\Account;
use Illuminate\View\View;
use Illuminate\Http\Request;
use App\Models\AccountTransaction;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\AccountTransactionStoreRequest;
use App\Http\Requests\AccountTransactionUpdateRequest;

class AccountTransactionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', AccountTransaction::class);

        $search = $request->get('search', '');

        $accountTransactions = AccountTransaction::search($search)
            ->latest()
            ->paginate(50)
            ->withQueryString();

        return view(
            'app.account_transactions.index',
            compact('accountTransactions', 'search')
        );
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', AccountTransaction::class);

        $accounts = Account::pluck('account_title', 'id');

        return view('app.account_transactions.create', compact('accounts'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(
        AccountTransactionStoreRequest $request
    ): RedirectResponse {
        $this->authorize('create', AccountTransaction::class);

        $validated = $request->validated();

        $accountTransaction = AccountTransaction::create($validated);

        return redirect()
            ->route('account-transactions.edit', $accountTransaction)
            ->withSuccess(__('crud.common.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(
        Request $request,
        AccountTransaction $accountTransaction
    ): View {
        $this->authorize('view', $accountTransaction);

        return view(
            'app.account_transactions.show',
            compact('accountTransaction')
        );
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(
        Request $request,
        AccountTransaction $accountTransaction
    ): View {
        $this->authorize('update', $accountTransaction);

        $accounts = Account::pluck('account_title', 'id');

        return view(
            'app.account_transactions.edit',
            compact('accountTransaction', 'accounts')
        );
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        AccountTransactionUpdateRequest $request,
        AccountTransaction $accountTransaction
    ): RedirectResponse {
        $this->authorize('update', $accountTransaction);

        $validated = $request->validated();

        $accountTransaction->update($validated);

        return redirect()
            ->route('account-transactions.edit', $accountTransaction)
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        AccountTransaction $accountTransaction
    ): RedirectResponse {
        $this->authorize('delete', $accountTransaction);

        try {
            // Delete related fees first
            $accountTransaction->fees()->delete();
            
            // Delete related files if any
            if (method_exists($accountTransaction, 'files')) {
                foreach ($accountTransaction->files as $file) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($file->file_path);
                    $file->delete();
                }
            }

            $accountTransaction->delete();

            return redirect()
                ->route('account-transactions.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('account-transactions.index')
                    ->withErrors(['error' => 'Cannot delete transaction. This transaction has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('account-transactions.index')
                ->withErrors(['error' => 'An error occurred while deleting the transaction. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('account-transactions.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
