<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Customer;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use App\Http\Requests\AccountStoreRequest;
use App\Http\Requests\AccountUpdateRequest;
use Illuminate\Support\Facades\Session;

class AccountController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): View
    {
        $this->authorize('view-any', Account::class);

        $search = $request->get('search', '');

        $accounts = Account::search($search)
            ->latest()
            ->where('customer_id', '!=',1)
            ->paginate(50)
            ->withQueryString();

        return view('app.accounts.index', compact('accounts', 'search'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request): View
    {
        $this->authorize('create', Account::class);

        $customers = Customer::pluck('customer_name', 'id');

        return view('app.accounts.create', compact('customers'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AccountStoreRequest $request): RedirectResponse
    {
        $this->authorize('create', Account::class);

        $validated = $request->validated();

        $account = Account::create($validated);

        return redirect()
            ->route('accounts.edit', $account)
            ->withSuccess(__('crud.common.created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Account $account): View
    {
        $this->authorize('view', $account);

        return view('app.accounts.show', compact('account'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request, Account $account): View
    {
        $this->authorize('update', $account);

        $customers = Customer::pluck('customer_name', 'id');

        return view('app.accounts.edit', compact('account', 'customers'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        AccountUpdateRequest $request,
        Account $account
    ): RedirectResponse {
        $this->authorize('update', $account);

        $validated = $request->validated();

        $account->update($validated);

        return redirect()
            ->route('accounts.edit', $account)
            ->withSuccess(__('crud.common.saved'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        Request $request,
        Account $account
    ): RedirectResponse {
        $this->authorize('delete', $account);

        try {
            // Check for related account transactions
            $hasTransactions = \App\Models\AccountTransaction::where('from_account_id', $account->id)
                ->orWhere('to_account_id', $account->id)
                ->exists();

            if ($hasTransactions) {
                return redirect()
                    ->route('accounts.index')
                    ->withErrors(['error' => 'Cannot delete account. This account has associated transactions.']);
            }

            $account->delete();

            return redirect()
                ->route('accounts.index')
                ->withSuccess(__('crud.common.removed'));
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                return redirect()
                    ->route('accounts.index')
                    ->withErrors(['error' => 'Cannot delete account. This account has related data that prevents deletion.']);
            }
            
            return redirect()
                ->route('accounts.index')
                ->withErrors(['error' => 'An error occurred while deleting the account. Please try again.']);
        } catch (\Exception $e) {
            return redirect()
                ->route('accounts.index')
                ->withErrors(['error' => 'An unexpected error occurred. Please try again.']);
        }
    }
}
