<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Color;

class TransactionsExport implements FromCollection, WithHeadings, WithMapping, WithStyles, WithColumnWidths, WithTitle, ShouldAutoSize
{
    protected $transactions;
    protected $account;
    protected $startDate;
    protected $endDate;
    protected $totals;

    public function __construct($transactions, $account, $startDate, $endDate, $totals)
    {
        $this->transactions = $transactions;
        $this->account = $account;
        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->totals = $totals;
    }

    public function collection()
    {
        // Convert array to collection if needed
        if (is_array($this->transactions)) {
            return collect($this->transactions);
        }
        return $this->transactions;
    }

    public function headings(): array
    {
        return [
            'Transaction Report',
            '',
            '',
            '',
            '',
            '',
            '',
            '',
        ];
    }

    public function map($transaction): array
    {
        // This method is not used since we're handling everything in styles()
        // But we need to return something to satisfy the interface
        return [
            'id' => $transaction['id'] ?? $transaction->id ?? 'N/A',
            'transaction_type' => $transaction['transaction_type'] ?? $transaction->transaction_type ?? 'N/A',
            'transaction_amount' => $transaction['transaction_amount'] ?? $transaction->transaction_amount ?? 0,
            'created_at' => $transaction['created_at'] ?? $transaction->created_at ?? 'N/A',
            'note' => $transaction['note'] ?? $transaction->note ?? '-',
        ];
    }

    public function title(): string
    {
        return 'Transactions Report';
    }

    public function columnWidths(): array
    {
        return [
            'A' => 15,  // #
            'B' => 20,  // Type
            'C' => 15,  // ID
            'D' => 15,  // Date
            'E' => 20,  // In
            'F' => 20,  // Out
            'G' => 20,  // Balance
            'H' => 30,  // Note
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Set title
        $sheet->mergeCells('A1:H1');
        $sheet->setCellValue('A1', 'LA DERMA - Transaction Report');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Set company info
        $sheet->mergeCells('A2:H2');
        $sheet->setCellValue('A2', 'Berat for Clinic');
        $sheet->getStyle('A2')->getFont()->setSize(12);
        $sheet->getStyle('A2')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Set date range
        $sheet->mergeCells('A3:H3');
        $sheet->setCellValue('A3', 'Period: ' . $this->startDate . ' to ' . $this->endDate);
        $sheet->getStyle('A3')->getFont()->setSize(11);
        $sheet->getStyle('A3')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Set account info
        $sheet->mergeCells('A4:H4');
        $sheet->setCellValue('A4', 'Customer: ' . $this->account['customer_name'] . ' (ID: ' . $this->account['customer_id'] . ') | Currency: ' . $this->account['currency_name'] . ' | Balance: ' . number_format($this->account['balance'], 0, '.', ','));
        $sheet->getStyle('A4')->getFont()->setSize(11);
        $sheet->getStyle('A4')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Empty row
        $sheet->setCellValue('A5', '');

        // Set table headers
        $sheet->setCellValue('A6', '#');
        $sheet->setCellValue('B6', 'Type');
        $sheet->setCellValue('C6', 'ID');
        $sheet->setCellValue('D6', 'Date');
        $sheet->setCellValue('E6', 'In');
        $sheet->setCellValue('F6', 'Out');
        $sheet->setCellValue('G6', 'Balance');
        $sheet->setCellValue('H6', 'Note');

        // Style headers
        $headerStyle = [
            'font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => '2C7BE5']],
            'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
            'borders' => [
                'allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]
            ]
        ];
        $sheet->getStyle('A6:H6')->applyFromArray($headerStyle);

        // Add transaction data
        $row = 7;
        $runningTotal = 0;
        
        foreach ($this->transactions as $transaction) {
            // Calculate running total
            if ($transaction['transaction_type'] == 'withdrawal') {
                $runningTotal -= $transaction['transaction_amount'];
            } else {
                $runningTotal += $transaction['transaction_amount'];
            }

            $sheet->setCellValue('A' . $row, $row - 6);
            $sheet->setCellValue('B' . $row, ucfirst($transaction['transaction_type']));
            $sheet->setCellValue('C' . $row, $transaction['id']);
            $sheet->setCellValue('D' . $row, $transaction['created_at']);
            
            // In column
            if ($transaction['transaction_type'] == 'deposit') {
                $sheet->setCellValue('E' . $row, number_format($transaction['transaction_amount'], 0, '.', ','));
            } else {
                $sheet->setCellValue('E' . $row, '0');
            }
            
            // Out column
            if ($transaction['transaction_type'] == 'withdrawal') {
                $sheet->setCellValue('F' . $row, number_format($transaction['transaction_amount'], 0, '.', ','));
            } else {
                $sheet->setCellValue('F' . $row, '0');
            }
            
            $sheet->setCellValue('G' . $row, number_format($runningTotal, 0, '.', ','));
            $sheet->setCellValue('H' . $row, $transaction['note'] ?? '-');

            // Style transaction rows
            $transactionStyle = [
                'borders' => [
                    'allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]
                ],
                'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER]
            ];
            $sheet->getStyle('A' . $row . ':H' . $row)->applyFromArray($transactionStyle);

            // Color coding for transaction types
            if ($transaction['transaction_type'] == 'deposit') {
                $sheet->getStyle('B' . $row)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('D4EDDA');
            } elseif ($transaction['transaction_type'] == 'withdrawal') {
                $sheet->getStyle('B' . $row)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('F8D7DA');
            } elseif ($transaction['transaction_type'] == 'transfer') {
                $sheet->getStyle('B' . $row)->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('D1ECF1');
            }

            $row++;
        }

        // Add totals row
        $totalsRow = $row;
        $sheet->setCellValue('A' . $totalsRow, '');
        $sheet->setCellValue('B' . $totalsRow, '');
        $sheet->setCellValue('C' . $totalsRow, '');
        $sheet->setCellValue('D' . $totalsRow, 'TOTAL');
        $sheet->setCellValue('E' . $totalsRow, number_format($this->totals['totalDeposit'], 0, '.', ','));
        $sheet->setCellValue('F' . $totalsRow, number_format($this->totals['totalWithdrawal'], 0, '.', ','));
        $sheet->setCellValue('G' . $totalsRow, number_format($this->totals['totalIncome'] - $this->totals['totalExpense'], 0, '.', ','));
        $sheet->setCellValue('H' . $totalsRow, '');

        // Style totals row
        $totalsStyle = [
            'font' => ['bold' => true],
            'fill' => ['fillType' => Fill::FILL_SOLID, 'startColor' => ['rgb' => 'F8F9FA']],
            'borders' => [
                'allBorders' => ['borderStyle' => Border::BORDER_THIN, 'color' => ['rgb' => '000000']]
            ],
            'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER]
        ];
        $sheet->getStyle('A' . $totalsRow . ':H' . $totalsRow)->applyFromArray($totalsStyle);

        // Add summary info
        $summaryRow = $totalsRow + 2;
        $sheet->setCellValue('A' . $summaryRow, 'Summary:');
        $sheet->getStyle('A' . $summaryRow)->getFont()->setBold(true);
        
        $summaryRow++;
        $sheet->setCellValue('A' . $summaryRow, 'Total Transactions:');
        $sheet->setCellValue('B' . $summaryRow, count($this->transactions));
        
        $summaryRow++;
        $sheet->setCellValue('A' . $summaryRow, 'Total Income:');
        $sheet->setCellValue('B' . $summaryRow, number_format($this->totals['totalIncome'], 0, '.', ',') . ' ' . $this->account['currency_name']);
        
        $summaryRow++;
        $sheet->setCellValue('A' . $summaryRow, 'Total Expense:');
        $sheet->setCellValue('B' . $summaryRow, number_format($this->totals['totalExpense'], 0, '.', ',') . ' ' . $this->account['currency_name']);
        
        $summaryRow++;
        $sheet->setCellValue('A' . $summaryRow, 'Net Balance:');
        $sheet->setCellValue('B' . $summaryRow, number_format($this->totals['totalIncome'] - $this->totals['totalExpense'], 0, '.', ',') . ' ' . $this->account['currency_name']);

        // Add footer
        $footerRow = $summaryRow + 2;
        $sheet->setCellValue('A' . $footerRow, 'Generated on: ' . now()->format('d/m/Y H:i:s'));
        $sheet->getStyle('A' . $footerRow)->getFont()->setSize(10)->setItalic(true);

        return $sheet;
    }
}
