<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\OneSignalService;
use App\Models\User;

class TestOneSignalConfig extends Command
{
    protected $signature = 'onesignal:test';
    protected $description = 'Test OneSignal configuration and authentication';

    public function handle()
    {
        $this->info('🔍 Testing OneSignal Configuration...');
        $this->line('');

        // Test 1: Check environment variables
        $this->info('1. Checking Environment Variables:');
        $appId = config('onesignal.app_id');
        $restApiKey = config('onesignal.rest_api_key');
        $userAuthKey = config('onesignal.user_auth_key');

        $this->line("   App ID: " . ($appId ? "✅ Set (length: " . strlen($appId) . ")" : "❌ Not Set"));
        $this->line("   REST API Key: " . ($restApiKey ? "✅ Set (length: " . strlen($restApiKey) . ")" : "❌ Not Set"));
        $this->line("   User Auth Key: " . ($userAuthKey ? "✅ Set (length: " . strlen($userAuthKey) . ")" : "❌ Not Set"));
        
        if ($restApiKey) {
            $this->line("   REST API Key Preview: " . substr($restApiKey, 0, 10) . "...");
        }
        $this->line('');

        // Test 2: Check database
        $this->info('2. Checking Database:');
        $usersWithPlayerIds = User::whereNotNull('onesignal_player_id')->count();
        $totalUsers = User::count();
        $this->line("   Users with OneSignal Player IDs: {$usersWithPlayerIds}");
        $this->line("   Total Users: {$totalUsers}");
        $this->line("   Coverage: " . ($totalUsers > 0 ? round(($usersWithPlayerIds / $totalUsers) * 100, 2) : 0) . "%");
        $this->line('');

        // Test 3: Test OneSignal Service
        $this->info('3. Testing OneSignal Service:');
        
        if (!$appId || !$restApiKey) {
            $this->error('   ❌ Cannot test OneSignal service - missing configuration');
            return;
        }

        try {
            $oneSignalService = new OneSignalService();
            
            // Test with a dummy player ID
            $testPlayerId = 'test-player-id-' . time();
            $this->line("   Testing with player ID: {$testPlayerId}");
            
            $translationService = new \App\Services\NotificationTranslationService();
            $testMessage = $translationService->getTestMessage();
            
            $result = $oneSignalService->sendNotificationToUsers(
                [$testPlayerId],
                $testMessage,
                ['type' => 'test']
            );
            
            if (is_array($result)) {
                if ($result['success'] === true) {
                    $this->info('   ✅ OneSignal service is working correctly!');
                    $this->line('   Response: ' . json_encode($result['data']));
                } else {
                    $this->error('   ❌ OneSignal service error:');
                    $this->line('   Error Type: ' . ($result['error_type'] ?? 'unknown'));
                    $this->line('   Error Message: ' . ($result['error'] ?? 'Unknown error'));
                    
                    if (isset($result['onesignal_errors'])) {
                        $this->line('   OneSignal API Errors: ' . json_encode($result['onesignal_errors']));
                    }
                    
                    if (isset($result['exception'])) {
                        $this->line('   Exception: ' . $result['exception']);
                    }
                }
            } else {
                $this->error('   ❌ OneSignal service returned unexpected response format');
            }
        } catch (\Exception $e) {
            $this->error('   ❌ Exception occurred: ' . $e->getMessage());
        }

        $this->line('');
        $this->info('4. Recommendations:');

        if (!$appId) {
            $this->line('   ❌ Add ONESIGNAL_APP_ID to your .env file');
        }

        if (!$restApiKey) {
            $this->line('   ❌ Add ONESIGNAL_REST_API_KEY to your .env file');
        }

        if ($usersWithPlayerIds === 0) {
            $this->line('   ⚠️  No users have OneSignal player IDs. Test the mobile app to generate player IDs.');
        }

        $this->line('');
        $this->info('✅ Test completed!');
    }
}


