<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class DebugOneSignalAuth extends Command
{
    protected $signature = 'onesignal:debug';
    protected $description = 'Debug OneSignal authentication issues';

    public function handle()
    {
        $this->info('🔍 Debugging OneSignal Authentication...');
        $this->line('');

        // Get configuration
        $appId = config('onesignal.app_id');
        $restApiKey = config('onesignal.rest_api_key');

        $this->info('Current Configuration:');
        $this->line("App ID: " . ($appId ? $appId : 'NOT SET'));
        $this->line("REST API Key: " . ($restApiKey ? substr($restApiKey, 0, 10) . '... (length: ' . strlen($restApiKey) . ')' : 'NOT SET'));
        $this->line('');

        if (!$appId || !$restApiKey) {
            $this->error('❌ Missing configuration. Please check your .env file.');
            return;
        }

        // Test 1: Check if the API key format looks correct
        $this->info('1. Checking API Key Format:');
        if (strlen($restApiKey) < 20) {
            $this->error('   ❌ API Key seems too short (should be 40+ characters)');
        } elseif (strlen($restApiKey) > 100) {
            $this->error('   ❌ API Key seems too long (should be around 40-60 characters)');
        } else {
            $this->line('   ✅ API Key length looks reasonable');
        }
        $this->line('');

        // Test 2: Make a simple API call to test authentication
        $this->info('2. Testing API Authentication:');
        
        try {
            $client = new Client([
                'base_uri' => 'https://onesignal.com/api/v1/',
                'headers' => [
                    'Authorization' => 'Basic ' . $restApiKey,
                    'Content-Type' => 'application/json',
                ],
            ]);

            // Try to get app info (this requires authentication)
            $response = $client->get('apps/' . $appId);
            $data = json_decode($response->getBody()->getContents(), true);
            
            $this->info('   ✅ Authentication successful!');
            $this->line('   App Name: ' . ($data['name'] ?? 'Unknown'));
            $this->line('   App ID: ' . ($data['id'] ?? 'Unknown'));
            
        } catch (GuzzleException $e) {
            $this->error('   ❌ Authentication failed!');
            $this->line('   Error: ' . $e->getMessage());
            
            if ($e->hasResponse()) {
                $response = $e->getResponse();
                $body = $response->getBody()->getContents();
                $this->line('   Response Body: ' . $body);
            }
        }

        $this->line('');

        // Test 3: Try sending a test notification
        $this->info('3. Testing Notification Send:');
        
        try {
            $payload = [
                'app_id' => $appId,
                'include_player_ids' => ['test-player-id-' . time()],
                'contents' => ['en' => 'Test notification from debug command'],
                'data' => ['type' => 'debug_test'],
            ];

            $this->line('   Sending test notification...');
            $response = $client->post('notifications', [
                'json' => $payload,
            ]);

            $responseData = json_decode($response->getBody()->getContents(), true);
            
            if (isset($responseData['errors'])) {
                $this->error('   ❌ OneSignal returned errors:');
                foreach ($responseData['errors'] as $error) {
                    $this->line('   - ' . $error);
                }
            } else {
                $this->info('   ✅ Test notification sent successfully!');
                $this->line('   Notification ID: ' . ($responseData['id'] ?? 'Unknown'));
            }
            
        } catch (GuzzleException $e) {
            $this->error('   ❌ Failed to send test notification');
            $this->line('   Error: ' . $e->getMessage());
            
            if ($e->hasResponse()) {
                $response = $e->getResponse();
                $body = $response->getBody()->getContents();
                $this->line('   Response Body: ' . $body);
            }
        }

        $this->line('');
        $this->info('4. Recommendations:');
        $this->line('   - Double-check your REST API Key from OneSignal dashboard');
        $this->line('   - Make sure you copied the entire key (no spaces, no quotes)');
        $this->line('   - Verify the key is from the correct app');
        $this->line('   - Run: php artisan config:clear && php artisan config:cache');
    }
}


