# Two-Factor Authentication (2FA) Setup Guide

## Overview
Two-Factor Authentication (2FA) with Google Authenticator has been successfully implemented in your Laravel application. This adds an extra layer of security to user accounts.

## Features Implemented

### 1. Database Setup
- ✅ Added `two_factor_secret` column to users table
- ✅ Added `two_factor_recovery_codes` column to users table
- ✅ Added `two_factor_confirmed_at` column to users table

### 2. User Model
- ✅ Added `TwoFactorAuthenticatable` trait from Laravel Fortify
- ✅ Hidden sensitive 2FA fields (secret and recovery codes) from API responses

### 3. Profile Page
- ✅ Enabled 2FA management section at `/user/profile`
- ✅ QR code generation for Google Authenticator
- ✅ Manual setup key display
- ✅ Recovery codes generation and display

### 4. Available Routes
The following routes are automatically registered:

| Method | Route | Purpose |
|--------|-------|---------|
| GET | `/user/profile` | View profile with 2FA settings |
| POST | `/user/two-factor-authentication` | Enable 2FA |
| DELETE | `/user/two-factor-authentication` | Disable 2FA |
| POST | `/user/confirmed-two-factor-authentication` | Confirm 2FA setup |
| GET | `/user/two-factor-qr-code` | Get QR code |
| GET | `/user/two-factor-recovery-codes` | Get recovery codes |
| POST | `/user/two-factor-recovery-codes` | Regenerate recovery codes |

## How to Use

### For Users

#### Enabling 2FA:
1. Navigate to **Profile Page**: `http://127.0.0.1:8000/user/profile`
2. Scroll to the **"Two Factor Authentication"** section
3. Click the **"Enable"** button
4. Enter your password when prompted
5. Scan the displayed QR code with Google Authenticator app:
   - iOS: Download from App Store
   - Android: Download from Google Play Store
6. Enter the 6-digit code from your authenticator app to confirm
7. **Important**: Save the recovery codes in a secure location!

#### Using 2FA During Login:
1. Enter your email and password as usual
2. On the next screen, enter the 6-digit code from Google Authenticator
3. Alternatively, you can use a recovery code if you don't have access to your authenticator app

#### Viewing Recovery Codes:
1. Go to your profile page
2. In the 2FA section, click **"Show Recovery Codes"**
3. Store these codes securely - each code can only be used once

#### Regenerating Recovery Codes:
1. Go to your profile page
2. Click **"Regenerate Recovery Codes"**
3. Your old codes will be invalidated
4. Save the new codes securely

#### Disabling 2FA:
1. Go to your profile page
2. Click **"Disable"** in the 2FA section
3. Confirm with your password

### For Developers

#### Configuration Files

**Fortify Config** (`config/fortify.php`):
```php
'features' => [
    Features::registration(),
    Features::resetPasswords(),
    Features::updateProfileInformation(),
    Features::updatePasswords(),
    Features::twoFactorAuthentication([
        'confirm' => true,
        'confirmPassword' => true,
    ]),
],
```

**User Model** (`app/Models/User.php`):
```php
use Laravel\Fortify\TwoFactorAuthenticatable;

class User extends Authenticatable
{
    use TwoFactorAuthenticatable;
    // ... other traits
}
```

#### Customization Options

1. **Window Configuration**: Adjust the time window for code validation
```php
Features::twoFactorAuthentication([
    'confirm' => true,
    'confirmPassword' => true,
    'window' => 1, // Allow codes from 1 window before/after current
]),
```

2. **Custom Views**: The views are located at:
   - `resources/views/profile/two-factor-authentication-form.blade.php`
   - `resources/views/auth/two-factor-challenge.blade.php`

3. **Recovery Codes**: By default, 8 recovery codes are generated. Each can be used once.

## Security Best Practices

### For Users:
1. ✅ Enable 2FA on all accounts with access to sensitive data
2. ✅ Store recovery codes in a password manager (NOT in plain text)
3. ✅ Never share your 2FA codes or recovery codes
4. ✅ If you lose access to your authenticator, use a recovery code immediately
5. ✅ Regenerate recovery codes if you suspect they've been compromised

### For Administrators:
1. ✅ Encourage or require 2FA for all users
2. ✅ Implement password confirmation before enabling/disabling 2FA
3. ✅ Log 2FA enable/disable events for security auditing
4. ✅ Consider implementing 2FA requirement for specific roles

## Troubleshooting

### "Invalid authentication code" error:
- **Cause**: Time sync issue between server and device
- **Solution**: 
  - Ensure server time is accurate (NTP sync)
  - Check device time is set to automatic
  - Try using a recovery code instead

### Lost access to authenticator app:
- **Solution**: Use one of your recovery codes to log in, then:
  1. Disable 2FA
  2. Re-enable 2FA with a new QR code
  3. Save new recovery codes

### QR code not displaying:
- **Cause**: View cache or configuration cache
- **Solution**: Run the following commands:
```bash
php artisan view:clear
php artisan config:cache
```

## Testing

To test the 2FA implementation:

1. **Test Enabling 2FA**:
   ```bash
   php artisan test --filter TwoFactorAuthenticationSettingsTest
   ```

2. **Manual Testing**:
   - Create a test user account
   - Enable 2FA from profile page
   - Log out and log back in
   - Verify 2FA challenge appears
   - Test with both authenticator code and recovery code

## Technical Details

### How it Works:
1. When enabled, a secret key is generated and encrypted in the database
2. This secret is used to generate time-based one-time passwords (TOTP)
3. Google Authenticator uses the same secret to generate matching codes
4. Codes are valid for 30 seconds and can be used within a configurable window
5. Recovery codes are one-time use tokens stored encrypted in the database

### Security Features:
- ✅ Password confirmation required for 2FA changes
- ✅ Secrets are encrypted in database
- ✅ Recovery codes are encrypted and hashed
- ✅ Rate limiting on 2FA attempts
- ✅ Session management for authenticated users

## Additional Resources

- [Laravel Fortify Documentation](https://laravel.com/docs/10.x/fortify)
- [Google Authenticator](https://support.google.com/accounts/answer/1066447)
- [RFC 6238 - TOTP Algorithm](https://tools.ietf.org/html/rfc6238)

## Support

If you encounter any issues with 2FA:
1. Check the application logs: `storage/logs/laravel.log`
2. Verify database migrations ran successfully
3. Ensure Fortify is properly configured
4. Check that the User model has the `TwoFactorAuthenticatable` trait

---

**Last Updated**: October 24, 2025
**Status**: ✅ Fully Implemented and Ready to Use

