# LA DERMA Production Deployment Guide

## 🚀 Production Deployment Checklist

### 1. Server Requirements
- **PHP**: 8.1 or higher
- **MySQL**: 5.7 or higher
- **Redis**: 6.0 or higher (for caching and sessions)
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **SSL Certificate**: Required for HTTPS

### 2. Environment Setup

#### Step 1: Clone Repository
```bash
git clone https://github.com/yourusername/cash-nrx.git
cd cash-nrx
```

#### Step 2: Install Dependencies
```bash
composer install --no-dev --optimize-autoloader
```

#### Step 3: Environment Configuration
```bash
# Copy production template
cp production.env.template .env

# Edit .env file with your production values
nano .env
```

**Required Environment Variables:**
```env
APP_NAME="LA DERMA"
APP_ENV=production
APP_DEBUG=false
APP_URL=https://LA DERMA.innovation-pulsehub.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=LA DERMA_production
DB_USERNAME=your_db_user
DB_PASSWORD=your_db_password

CACHE_DRIVER=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis

SANCTUM_STATEFUL_DOMAINS=LA DERMA.innovation-pulsehub.com,innovation-pulsehub.com
```

#### Step 4: Generate Application Key
```bash
php artisan key:generate
```

### 3. Database Setup

#### Step 1: Create Database
```sql
CREATE DATABASE LA DERMA_production CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
CREATE USER 'LA DERMA_user'@'localhost' IDENTIFIED BY 'secure_password';
GRANT ALL PRIVILEGES ON LA DERMA_production.* TO 'LA DERMA_user'@'localhost';
FLUSH PRIVILEGES;
```

#### Step 2: Run Migrations
```bash
php artisan migrate --force
```

#### Step 3: Seed Initial Data
```bash
php artisan db:seed --force
```

### 4. Application Optimization

#### Step 1: Clear and Cache Configurations
```bash
php artisan config:clear
php artisan config:cache
php artisan route:clear
php artisan route:cache
php artisan view:clear
php artisan view:cache
```

#### Step 2: Set Permissions
```bash
chmod -R 755 storage
chmod -R 755 bootstrap/cache
chown -R www-data:www-data storage
chown -R www-data:www-data bootstrap/cache
```

#### Step 3: Create Storage Symlink
```bash
php artisan storage:link
```

### 5. Web Server Configuration

#### Apache Configuration (.htaccess)
```apache
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^(.*)$ index.php [QSA,L]
</IfModule>

# Security Headers
<IfModule mod_headers.c>
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
    Header always set Permissions-Policy "geolocation=(), microphone=(), camera=()"
</IfModule>
```

#### Nginx Configuration
```nginx
server {
    listen 80;
    listen 443 ssl http2;
    server_name LA DERMA.innovation-pulsehub.com;
    root /path/to/cash-nrx/public;

    # SSL Configuration
    ssl_certificate /path/to/ssl/certificate.crt;
    ssl_certificate_key /path/to/ssl/private.key;

    # Security Headers
    add_header X-Content-Type-Options nosniff;
    add_header X-Frame-Options DENY;
    add_header X-XSS-Protection "1; mode=block";
    add_header Referrer-Policy "strict-origin-when-cross-origin";

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.(?!well-known).* {
        deny all;
    }
}
```

### 6. SSL Certificate Setup

#### Using Let's Encrypt (Certbot)
```bash
# Install Certbot
sudo apt install certbot python3-certbot-apache

# Get SSL Certificate
sudo certbot --apache -d LA DERMA.innovation-pulsehub.com

# Auto-renewal
sudo crontab -e
# Add: 0 12 * * * /usr/bin/certbot renew --quiet
```

### 7. Redis Configuration

#### Install Redis
```bash
sudo apt install redis-server
sudo systemctl enable redis-server
sudo systemctl start redis-server
```

#### Configure Redis
```bash
sudo nano /etc/redis/redis.conf
```

**Important Settings:**
```
bind 127.0.0.1
port 6379
requirepass your_redis_password
maxmemory 256mb
maxmemory-policy allkeys-lru
```

### 8. Monitoring and Logging

#### Set up Log Rotation
```bash
sudo nano /etc/logrotate.d/LA DERMA
```

**Configuration:**
```
/path/to/LA DERMA/storage/logs/*.log {
    daily
    missingok
    rotate 14
    compress
    notifempty
    create 644 www-data www-data
}
```

#### Set up Monitoring
```bash
# Install monitoring tools
sudo apt install htop iotop nethogs

# Set up uptime monitoring (using UptimeRobot or similar)
```

### 9. Backup Strategy

#### Database Backup Script
```bash
#!/bin/bash
# Create backup script: /path/to/backup.sh

DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/path/to/backups"
DB_NAME="LA DERMA_production"

# Create backup
mysqldump -u LA DERMA_user -p$DB_PASSWORD $DB_NAME > $BACKUP_DIR/LA DERMA_$DATE.sql

# Compress backup
gzip $BACKUP_DIR/LA DERMA_$DATE.sql

# Keep only last 7 days of backups
find $BACKUP_DIR -name "LA DERMA_*.sql.gz" -mtime +7 -delete
```

#### Set up Cron Job
```bash
# Add to crontab
0 2 * * * /path/to/backup.sh
```

### 10. API Testing

#### Test Authentication
```bash
curl -X POST https://LA DERMA.innovation-pulsehub.com/api/mobile/login \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@admin.com","password":"admin"}'
```

#### Test Protected Endpoint
```bash
curl -X GET https://LA DERMA.innovation-pulsehub.com/api/mobile/user \
  -H "Authorization: Bearer YOUR_TOKEN"
```

### 11. Flutter App Configuration

Update your Flutter app's API service to use production URL:

```dart
// In lib/services/api_service.dart
static const String baseUrl = 'https://LA DERMA.innovation-pulsehub.com/api/mobile';
```

### 12. Security Checklist

- [ ] HTTPS enabled
- [ ] SSL certificate valid
- [ ] Database credentials secure
- [ ] API rate limiting configured
- [ ] CORS properly configured
- [ ] Security headers set
- [ ] File permissions correct
- [ ] Error logging enabled
- [ ] Backup strategy in place
- [ ] Monitoring configured

### 13. Performance Optimization

#### Enable OPcache
```bash
# Edit php.ini
sudo nano /etc/php/8.1/apache2/php.ini

# Add/Update:
opcache.enable=1
opcache.memory_consumption=128
opcache.interned_strings_buffer=8
opcache.max_accelerated_files=4000
opcache.revalidate_freq=2
opcache.fast_shutdown=1
```

#### Enable Gzip Compression
```apache
# Add to .htaccess
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
</IfModule>
```

### 14. Troubleshooting

#### Common Issues

1. **500 Internal Server Error**
   - Check file permissions
   - Check Laravel logs: `tail -f storage/logs/laravel.log`
   - Verify .env configuration

2. **Database Connection Error**
   - Verify database credentials
   - Check MySQL service status
   - Test database connection

3. **CORS Issues**
   - Update CORS configuration
   - Check allowed origins
   - Verify middleware registration

4. **Token Issues**
   - Check Sanctum configuration
   - Verify token expiration
   - Check middleware order

### 15. Maintenance

#### Regular Tasks
- Monitor server resources
- Check application logs
- Update dependencies
- Test backup restoration
- Review security logs
- Update SSL certificates

#### Updates
```bash
# Update dependencies
composer update --no-dev

# Clear caches
php artisan config:clear
php artisan config:cache
php artisan route:clear
php artisan route:cache
```

## 🎉 Deployment Complete!

Your LA DERMA API is now ready for production! 

**API Endpoint:** `https://LA DERMA.innovation-pulsehub.com/api/mobile`
**Flutter App:** Update the base URL in your Flutter app
**Documentation:** Available at `API_DOCUMENTATION.md`

### Next Steps:
1. Test all API endpoints
2. Deploy your Flutter app
3. Set up monitoring alerts
4. Configure automated backups
5. Set up staging environment for testing
