# Kurdish Text Encoding Fix Instructions

## Problem
Kurdish text appears as "???? ???" in API responses because the data is corrupted in the database.

## Solution Overview
1. Update database charset to UTF-8
2. Fix corrupted data in the database
3. Apply code changes to prevent future corruption

---

## Step 1: Run Database Migration

Open your terminal and navigate to the Laravel project:

```bash
cd /Users/esramuhannad/development/cash-nrx
```

Run the migration:

```bash
php artisan migrate
```

If `php` is not found, try using the full path or MAMP/Herd/Docker PHP:

```bash
# For MAMP
/Applications/MAMP/bin/php/php8.2.0/bin/php artisan migrate

# For Herd
~/.config/herd/bin/php artisan migrate

# For Docker/Sail
./vendor/bin/sail artisan migrate
```

---

## Step 2: Fix Transaction 129518 (Known Corrupted Data)

### Option A: Using PHP Script

Run the fix script:

```bash
php fix_specific_kurdish_text.php
```

### Option B: Using Laravel Tinker

```bash
php artisan tinker
```

Then paste this code:

```php
DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');

$transaction = App\Models\AccountTransaction::find(129518);
if ($transaction) {
    $transaction->note = 'ئاسۆ کوی';
    
    // Disable model events to avoid double encoding
    App\Models\AccountTransaction::flushEventListeners();
    App\Models\AccountTransaction::boot();
    
    $transaction->save();
    echo "✓ Transaction 129518 fixed!\n";
    echo "Note: " . $transaction->fresh()->note . "\n";
} else {
    echo "✗ Transaction not found!\n";
}
```

### Option C: Direct Database Update

Connect to your MySQL database and run:

```sql
SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci;

UPDATE account_transactions 
SET note = 'ئاسۆ کوی' 
WHERE id = 129518;

SELECT id, note FROM account_transactions WHERE id = 129518;
```

---

## Step 3: Find and Fix Other Corrupted Transactions

### Find all corrupted transactions:

```bash
php artisan tinker
```

```php
$corrupted = App\Models\AccountTransaction::where(function($q) {
    $q->where('note', 'like', '%?%')
      ->orWhere('person_name', 'like', '%?%');
})->select('id', 'transaction_type', 'note', 'person_name', 'created_at')
  ->orderBy('id', 'desc')
  ->limit(20)
  ->get();

foreach ($corrupted as $t) {
    echo "ID: {$t->id} | Type: {$t->transaction_type} | Note: {$t->note} | Person: {$t->person_name}\n";
}
```

### Fix them individually (if you know the original text):

```php
DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');

App\Models\AccountTransaction::flushEventListeners();
App\Models\AccountTransaction::boot();

$transaction = App\Models\AccountTransaction::find(TRANSACTION_ID);
$transaction->note = 'ORIGINAL_KURDISH_TEXT';
$transaction->save();
```

---

## Step 4: Verify the Fix

Test the API endpoint in Postman:

```
GET https://LA DERMA.innovation-pulsehub.com/api/mobile/transactions
Authorization: Bearer YOUR_TOKEN
```

The response should now show:

```json
{
    "note": "ئاسۆ کوی"
}
```

Instead of:

```json
{
    "note": "???? ???"
}
```

---

## Step 5: Test Kurdish Text Encoding Endpoint

```
POST https://LA DERMA.innovation-pulsehub.com/api/mobile/test-kurdish
Content-Type: application/json

{
    "text": "ئاسۆ کوی"
}
```

Expected response:

```json
{
    "original": "ئاسۆ کوی",
    "stored_note": "ئاسۆ کوی",
    "stored_person_name": "ئاسۆ کوی",
    "note_matches": true,
    "person_name_matches": true,
    "note_encoding": "UTF-8",
    "person_name_encoding": "UTF-8"
}
```

---

## Files Changed

### Backend (Laravel)
1. ✅ `config/database.php` - Added UTF-8 PDO initialization
2. ✅ `app/Models/AccountTransaction.php` - Added automatic UTF-8 encoding in model events
3. ✅ `app/Http/Controllers/Api/TransactionController.php` - Added UTF-8 encoding + JSON_UNESCAPED_UNICODE
4. ✅ `database/migrations/2025_01_15_000000_fix_kurdish_text_encoding_in_account_transactions.php` - Database charset fix
5. ✅ `routes/api.php` - Added test endpoint

### Frontend (Flutter)
1. ✅ `lib/providers/theme_provider.dart` - Uses NotoSansArabic font
2. ✅ `lib/screens/transactions_tab.dart` - Uses GoogleFonts.notoSansArabic for descriptions
3. ✅ `lib/utils/kurdish_text_utils.dart` - Kurdish text detection utility

---

## Prevention for Future Data

All the code changes are already in place to prevent future corruption:

- ✅ Database uses UTF-8MB4 charset
- ✅ Model automatically encodes on create/update
- ✅ Controller encodes on retrieval
- ✅ API responses use JSON_UNESCAPED_UNICODE
- ✅ Frontend uses proper Kurdish fonts

Any NEW transactions created will automatically handle Kurdish text correctly. You only need to fix the EXISTING corrupted data.

---

## Quick Command Reference

```bash
# Navigate to project
cd /Users/esramuhannad/development/cash-nrx

# Run migration
php artisan migrate

# Fix specific transaction
php fix_specific_kurdish_text.php

# Or use tinker
php artisan tinker
# Then paste the fix code above

# Test the API
curl -X GET "https://LA DERMA.innovation-pulsehub.com/api/mobile/transactions" \
  -H "Authorization: Bearer YOUR_TOKEN"
```

---

## Troubleshooting

### If PHP is not found:
- Use MAMP: `/Applications/MAMP/bin/php/php8.2.0/bin/php`
- Use Herd: `~/.config/herd/bin/php`
- Use Docker: `./vendor/bin/sail php`

### If migration fails:
- Check database connection in `.env`
- Ensure MySQL/MariaDB is running
- Check database user permissions

### If text still shows as "???":
- The data is still corrupted in the database
- You must manually update it with the original text
- Use the SQL or Tinker methods above

---

## Need Help?

If you encounter any issues:
1. Check the Laravel logs: `storage/logs/laravel.log`
2. Check the migration status: `php artisan migrate:status`
3. Verify database charset: `SHOW CREATE TABLE account_transactions;`



