# Flutter OneSignal Integration Fix

## Current Status

✅ Currency rate notifications working  
❌ Transaction notifications failing

## The Problem

Transaction notifications fail with:
```
All included players are not subscribed
```

This happens because:
1. ✅ Backend has External IDs set (`onesignal_external_id = user_id`)
2. ❌ Mobile app hasn't told OneSignal about the External ID → Device connection

## The Solution

You need to call `OneSignal.shared.setExternalUserId()` in your Flutter app after user logs in.

---

## Implementation Steps

### Step 1: Find Your OneSignal Initialization Code

Look for where you initialize OneSignal in your Flutter app (probably in `main.dart` or auth service):

```dart
import 'package:onesignal_flutter/onesignal_flutter.dart';
```

### Step 2: Set External User ID After Login

In your login function (wherever you handle successful login), add:

```dart
Future<void> login(String email, String password) async {
  try {
    // Your existing login API call
    final response = await apiService.post('/mobile/login', {
      'email': email,
      'password': password,
    });
    
    final user = response['user'];
    final userId = user['id'];
    
    // Save user data, token, etc.
    // ... your existing code ...
    
    // 🔥 ADD THIS: Link OneSignal device to user ID
    await _setupOneSignalExternalId(userId);
    
  } catch (e) {
    // handle error
  }
}

Future<void> _setupOneSignalExternalId(int userId) async {
  try {
    // Set External User ID in OneSignal
    await OneSignal.shared.setExternalUserId(userId.toString());
    
    print('✅ OneSignal External User ID set: $userId');
    
    // Also send to backend API (optional but recommended)
    await apiService.post('/mobile/onesignal/external-id', {
      'external_id': userId.toString(),
    });
    
  } catch (e) {
    print('❌ Failed to set OneSignal External User ID: $e');
  }
}
```

### Step 3: Update Player ID Registration

Make sure your Player ID is also being sent to backend:

```dart
Future<void> initOneSignal() async {
  // Remove this method when app is killed on iOS
  OneSignal.shared.setAppId("YOUR_ONESIGNAL_APP_ID");

  // The promptForPushNotificationsWithUserResponse function will show the iOS push notification prompt.
  OneSignal.shared.promptUserForPushNotificationPermission().then((accepted) {
    print("Accepted permission: $accepted");
  });
  
  // Listen for subscription changes
  OneSignal.shared.setSubscriptionObserver((changes) async {
    print('OneSignal subscription changed: $changes');
    
    if (changes.to.userId != null) {
      // Send Player ID to backend
      await _sendPlayerIdToBackend(changes.to.userId!);
    }
  });
}

Future<void> _sendPlayerIdToBackend(String playerId) async {
  try {
    await apiService.post('/mobile/onesignal/player-id', {
      'player_id': playerId,
    });
    print('✅ Player ID sent to backend: $playerId');
  } catch (e) {
    print('❌ Failed to send Player ID to backend: $e');
  }
}
```

### Step 4: Handle Logout

When user logs out, remove the External User ID:

```dart
Future<void> logout() async {
  try {
    // Your existing logout logic
    // ...
    
    // Remove OneSignal External User ID
    await OneSignal.shared.removeExternalUserId();
    
    print('✅ OneSignal External User ID removed');
  } catch (e) {
    print('❌ Failed to remove OneSignal External User ID: $e');
  }
}
```

---

## Complete Example

Here's a complete example of an auth service with OneSignal:

```dart
import 'package:onesignal_flutter/onesignal_flutter.dart';

class AuthService {
  final ApiService apiService;
  
  AuthService(this.apiService);

  // Initialize OneSignal (call this in main.dart or app startup)
  Future<void> initOneSignal() async {
    OneSignal.shared.setAppId("YOUR_ONESIGNAL_APP_ID");
    
    // Request notification permission
    OneSignal.shared.promptUserForPushNotificationPermission();
    
    // Listen for subscription changes
    OneSignal.shared.setSubscriptionObserver((changes) async {
      if (changes.to.userId != null) {
        await _sendPlayerIdToBackend(changes.to.userId!);
      }
    });
    
    // Listen for notification clicks
    OneSignal.shared.setNotificationOpenedHandler((notification) {
      print('Notification opened: ${notification.notification.jsonRepresentation()}');
      // Handle notification tap (navigate to transaction detail, etc.)
    });
  }

  Future<void> login(String email, String password) async {
    try {
      final response = await apiService.post('/mobile/login', {
        'email': email,
        'password': password,
      });
      
      final user = response['user'];
      final token = response['token'];
      final userId = user['id'];
      
      // Save token and user data
      await _saveAuthData(token, user);
      
      // 🔥 CRITICAL: Link OneSignal device to user
      await _setupOneSignalExternalId(userId);
      
    } catch (e) {
      rethrow;
    }
  }

  Future<void> _setupOneSignalExternalId(int userId) async {
    try {
      // Set in OneSignal
      await OneSignal.shared.setExternalUserId(userId.toString());
      print('✅ OneSignal External User ID set: $userId');
      
      // Send to backend
      try {
        await apiService.post('/mobile/onesignal/external-id', {
          'external_id': userId.toString(),
        });
      } catch (e) {
        print('⚠️ Failed to update backend External ID: $e');
        // Don't fail the login if backend update fails
      }
    } catch (e) {
      print('❌ Failed to set OneSignal External User ID: $e');
    }
  }

  Future<void> _sendPlayerIdToBackend(String playerId) async {
    try {
      await apiService.post('/mobile/onesignal/player-id', {
        'player_id': playerId,
      });
      print('✅ Player ID sent to backend');
    } catch (e) {
      print('⚠️ Failed to send Player ID to backend: $e');
    }
  }

  Future<void> logout() async {
    try {
      // Call logout API
      await apiService.post('/mobile/logout');
      
      // Remove OneSignal association
      await OneSignal.shared.removeExternalUserId();
      
      // Clear local data
      await _clearAuthData();
      
    } catch (e) {
      rethrow;
    }
  }
}
```

---

## Testing After Implementation

### 1. Test with One User:

1. User logs out completely from app
2. User logs back in
3. Check backend: `php check_user_onesignal.php [user_email]`
4. Should show External ID is set
5. Create a transaction for that user
6. User should receive notification

### 2. Check Logs:

```bash
# Backend logs
tail -f storage/logs/laravel.log | grep -i 'transaction notification'

# Should see successful notification sends
```

### 3. Test Script:

```bash
php test_transaction_notification.php [user_email]
```

Should show:
```
✅ SUCCESS: Notification sent!
```

---

## Why Currency Rates Work But Transactions Don't

If currency rates are working, it's likely because:

1. Currency rate notifications might be using a different sending method
2. OR they were tested before External ID was properly set
3. OR there's a race condition where External ID isn't set yet when transaction is created

The fix above ensures External ID is ALWAYS set right after login, so both types of notifications will work.

---

## Debugging

If it still doesn't work after implementation:

### Check in App:
```dart
// Add this to check status
final status = await OneSignal.shared.getDeviceState();
print('OneSignal Status:');
print('  User ID (Player ID): ${status?.userId}');
print('  Push Token: ${status?.pushToken}');
print('  Subscribed: ${status?.isSubscribed}');

// Check external ID (Flutter OneSignal doesn't have a getter, so just log after setting)
```

### Check Backend:
```bash
php test_transaction_notification.php [user_email]
```

### Check OneSignal Dashboard:
1. Go to https://app.onesignal.com
2. Select your app
3. Go to "Audience" → "All Users"
4. Search for the user's External ID
5. Should see the device listed

---

## Summary

✅ **Transaction notification code is already correct** - it uses language preferences  
✅ **Backend is configured properly**  
❌ **Mobile app needs to set External User ID** - Add the code above

After implementing this fix:
- Users will receive transaction notifications in their preferred language
- Both currency rate and transaction notifications will work
- Notifications will be properly delivered to the correct devices













