# Currency Rate Access Implementation

## Overview
Added a checkbox in the Customer Management page to control the `currency_rate_access` column for each customer's associated user account.

## Changes Made

### 1. Database
- **Column**: `currency_rate_access` in `users` table
- **Type**: `boolean` (tinyint(1))
- **Default**: `false` (0)
- **Migration**: `database/migrations/2025_01_05_000000_add_currency_rate_access_to_users_table.php`

### 2. Model Updates
**File**: `app/Models/User.php`
- Column added to `$fillable` array (line 34)
- Column added to `$casts` array as boolean (line 49)

### 3. View Updates
**File**: `resources/views/livewire/customer-management.blade.php`
- Added new table column header: "Currency Rate Access"
- Added toggle switch checkbox for each customer row
- Shows "Enabled" (green badge) or "Disabled" (gray badge)
- Displays "-" if customer has no associated user

**Features**:
- Bootstrap switch-style checkbox
- Visual feedback with colored badges
- Only shows for customers with associated user accounts

### 4. Livewire Component Updates
**File**: `app/Livewire/CustomerManagement.php`
- Added `toggleCurrencyRateAccess($customerId)` method
- Method toggles the value between 0 and 1
- Includes error handling and logging
- Shows success/error alerts to user
- Updated `render()` method to eager load `user` relationship

## How It Works

1. **Display**: The checkbox shows the current state of `currency_rate_access` for each customer's user
2. **Toggle**: Clicking the checkbox calls `toggleCurrencyRateAccess()` via Livewire
3. **Update**: The method updates the user's `currency_rate_access` value
4. **Feedback**: A success message shows whether access was enabled or disabled
5. **Logging**: All actions are logged for debugging

## Testing Instructions

### 1. Verify Migration
```bash
php artisan migrate:status
```
Make sure `2025_01_05_000000_add_currency_rate_access_to_users_table` is migrated.

If not, run:
```bash
php artisan migrate
```

### 2. Test the Checkbox
1. Navigate to the Customer Management page
2. Find a customer in the list
3. Click the toggle switch in the "Currency Rate Access" column
4. You should see:
   - The checkbox toggle state change
   - A success alert message
   - The badge change from "Disabled" to "Enabled" (or vice versa)

### 3. Verify Database Update
```bash
php artisan tinker
```

Then run:
```php
// Check a specific user's currency_rate_access
$user = App\Models\User::find(USER_ID);
echo $user->currency_rate_access;

// List all users with currency rate access enabled
App\Models\User::where('currency_rate_access', 1)->get(['id', 'name', 'currency_rate_access']);
```

### 4. Check Logs
If there are any issues, check the Laravel log:
```bash
# Windows PowerShell
Get-Content storage/logs/laravel.log -Tail 50

# Linux/Mac
tail -f storage/logs/laravel.log
```

Look for entries like:
- `Toggling currency_rate_access for user X`
- `Verified currency_rate_access for user X`

## Troubleshooting

### Checkbox doesn't toggle
1. Check browser console for JavaScript errors
2. Make sure Livewire is loaded properly
3. Verify the customer has an associated user account

### Value doesn't save
1. Check Laravel logs for errors
2. Verify `currency_rate_access` is in User model's `$fillable` array
3. Check database permissions
4. Run: `php artisan config:clear && php artisan cache:clear`

### Migration not found
Run:
```bash
php artisan migrate:fresh --seed
```
**Warning**: This will reset your database!

## Code Locations

- **Migration**: `database/migrations/2025_01_05_000000_add_currency_rate_access_to_users_table.php`
- **User Model**: `app/Models/User.php` (lines 34, 49)
- **View**: `resources/views/livewire/customer-management.blade.php` (lines 49, 75-97)
- **Component**: `app/Livewire/CustomerManagement.php` (lines 261-308)

## API/Usage

To check or set currency rate access programmatically:

```php
// Get customer with user
$customer = Customer::with('user')->find($customerId);

// Check access
if ($customer->user && $customer->user->currency_rate_access) {
    // User has currency rate access
}

// Enable access
$customer->user->update(['currency_rate_access' => 1]);

// Disable access
$customer->user->update(['currency_rate_access' => 0]);

// Toggle access
$customer->user->currency_rate_access = !$customer->user->currency_rate_access;
$customer->user->save();
```

