# 🔄 Automatic Kurdish Text Fix - Documentation

## Overview

This system automatically detects and fixes corrupted Kurdish text **before returning it via the API**. You don't need to manually fix the database - the API will handle it on-the-fly!

---

## 🎯 How It Works

### 1. **Detection**
When the API retrieves transactions, it checks if the `note` or `person_name` fields contain corrupted text (characters like "?").

### 2. **Automatic Fix**
If corrupted text is detected:
- ✅ Check if there's a known mapping in `config/kurdish_mappings.php`
- ✅ If found, replace with the correct Kurdish text
- ✅ If not found, try different character encodings
- ✅ Return the fixed text in the API response

### 3. **No Database Change**
- ❌ The corrupted data **stays in the database** (not modified)
- ✓ The API **returns the correct text** to the frontend
- ✓ New transactions are saved with proper encoding

---

## 📁 Files Created

### 1. **`app/Helpers/KurdishTextHelper.php`**
The main helper class that:
- Detects corrupted text
- Fixes known corruption patterns
- Tries different encodings
- Ensures UTF-8 encoding

### 2. **`config/kurdish_mappings.php`**
Configuration file where you can add mappings for corrupted text:

```php
'mappings' => [
    '???? ???' => 'ئاسۆ کوی',  // Known corruption
    // Add more mappings here as you find them
],
```

### 3. **Modified `app/Http/Controllers/Api/TransactionController.php`**
Updated to use `KurdishTextHelper::process()` on all text fields before returning them.

---

## 🚀 How to Use

### No Action Required for Basic Use!
The system is **already active**. Just test your API:

```http
GET /api/mobile/transactions
Authorization: Bearer YOUR_TOKEN
```

The response will automatically show fixed Kurdish text:
```json
{
    "note": "ئاسۆ کوی"  // ✓ Not "???? ???"
}
```

---

## ➕ Adding New Corrupted Text Mappings

When you discover new corrupted text in the database:

### Step 1: Find the Original Text
You need to know what the original Kurdish text was supposed to be.

### Step 2: Add to Config
Edit `config/kurdish_mappings.php`:

```php
'mappings' => [
    '???? ???' => 'ئاسۆ کوی',      // Existing mapping
    '??? ?????' => 'کارتی Qi',      // Add new mapping
    '?? ???' => 'ڕێنماییەکان',     // Add another
],
```

### Step 3: Clear Config Cache
```bash
php artisan config:clear
```

### Step 4: Test
The API will now automatically fix this corrupted text!

---

## 🧪 Testing

### Test the Helper Directly
```bash
php test_kurdish_helper.php
```

### Test the API
```http
GET /api/mobile/transactions
```

Check if transaction 129518 now shows correct Kurdish text.

### Test with Specific Transaction
```http
GET /api/mobile/transactions/129518/details
```

---

## ⚙️ Configuration Options

### `config/kurdish_mappings.php`

```php
return [
    // Known corrupted patterns and their fixes
    'mappings' => [
        '???? ???' => 'ئاسۆ کوی',
    ],

    // Enable/disable auto-fix
    'auto_fix_enabled' => true,
    
    // What to return for unfixable corrupted text
    // Options: 'null', 'empty', 'original'
    'unfixable_return' => 'null',
    
    // Encodings to try when fixing
    'try_encodings' => [
        'Windows-1256',
        'ISO-8859-6',
        'CP1256',
    ],
];
```

---

## 📊 Handling Different Scenarios

### Scenario 1: Known Corrupted Text
**Database:** `"???? ???"`  
**Config Mapping:** `'???? ???' => 'ئاسۆ کوی'`  
**API Returns:** `"ئاسۆ کوی"` ✓

### Scenario 2: Unknown Corrupted Text
**Database:** `"??? ?????"`  
**Config Mapping:** None  
**API Returns:** `null` (or empty/original based on config)

### Scenario 3: Clean Kurdish Text
**Database:** `"ڕاکێشان"`  
**API Returns:** `"ڕاکێشان"` ✓ (unchanged)

### Scenario 4: New Transaction
**Frontend Sends:** `"ئاسۆ کوی"`  
**Database Stores:** `"ئاسۆ کوی"` (proper UTF-8)  
**API Returns:** `"ئاسۆ کوی"` ✓

---

## 🔍 Finding Corrupted Transactions

### Find all corrupted transactions:

```bash
php artisan tinker
```

```php
use App\Models\AccountTransaction;

$corrupted = AccountTransaction::where(function($q) {
    $q->where('note', 'like', '%?%')
      ->orWhere('person_name', 'like', '%?%');
})->select('id', 'note', 'person_name', 'created_at')
  ->orderBy('id', 'desc')
  ->get();

foreach ($corrupted as $t) {
    echo "ID: {$t->id}\n";
    echo "Note: {$t->note}\n";
    echo "Person: {$t->person_name}\n";
    echo "---\n";
}
```

---

## ✅ Advantages of This Approach

1. **No Database Modification** - Original data stays intact
2. **Automatic** - No manual intervention needed
3. **Configurable** - Easy to add new mappings
4. **Safe** - Unknown corrupted text returns null instead of wrong data
5. **Future-Proof** - New transactions save with proper encoding
6. **Flexible** - Can disable or change behavior via config

---

## 🆘 Troubleshooting

### Problem: API still returns "????"
**Solution 1:** Clear config cache
```bash
php artisan config:clear
php artisan cache:clear
```

**Solution 2:** Check the exact corrupted text
```php
// In tinker
$t = \App\Models\AccountTransaction::find(129518);
echo "'{$t->note}'"; // Check exact format
```

**Solution 3:** Add exact match to config
```php
// In config/kurdish_mappings.php
'???? ???' => 'ئاسۆ کوی',  // Exact match
```

### Problem: Returns null instead of text
**Solution:** The text is corrupted and has no mapping. Add it to config:
```php
'YOUR_CORRUPTED_TEXT' => 'CORRECT_KURDISH_TEXT',
```

### Problem: New transactions still corrupted
**Solution:** Check database charset
```sql
SHOW CREATE TABLE account_transactions;
```
Should show `utf8mb4` and `utf8mb4_unicode_ci`.

---

## 📝 Summary

### What Happens Now:

1. **Existing Corrupted Data** → API automatically fixes using mappings
2. **Unknown Corrupted Data** → API returns null (can be configured)
3. **Clean Data** → Passes through unchanged
4. **New Transactions** → Saved with proper UTF-8 encoding

### You Need To:

1. ✅ Test the API - it should already work
2. ✅ Add more mappings as you discover corrupted text
3. ✅ Share the original Kurdish text when you find corrupted data

### You DON'T Need To:

- ❌ Manually fix the database
- ❌ Run migration scripts repeatedly
- ❌ Worry about future transactions

---

## 🎉 Result

Your API now automatically handles Kurdish text corruption and returns proper Kurdish characters to your Flutter app!

```json
{
    "id": 129518,
    "note": "ئاسۆ کوی"  // ✓ Beautiful Kurdish text!
}
```



