# App Version Update System - Setup & Usage Guide

## 🔍 Problem Analysis

You experienced two main issues:
1. **404 Error in Postman**: The endpoint `/api/mobile/app-version` returns 404
2. **"Cannot Connect" when clicking Update**: The app shows the update dialog but fails to open the store

## ✅ Solutions Implemented

### 1. Backend Fixes

#### Issue: Empty `app_versions` table
The controller returns 404 when there's no version data in the database.

#### Solution: Seed the database with version data

**Option A: Use the Seeder (Recommended)**
```bash
cd /Users/esramuhannad/Development/cash-nrx

# Run the seeder
php artisan db:seed --class=AppVersionSeeder
```

**Option B: Use SQL Script**
```bash
# Execute the SQL file in your database
mysql -u your_username -p your_database < INSERT_APP_VERSION.sql

# Or run it through phpMyAdmin, TablePlus, etc.
```

**Option C: Manual Insert via Laravel Tinker**
```bash
php artisan tinker

# Then run this PHP code:
\App\Models\AppVersion::create([
    'version' => '1.0.15',
    'version_code' => 15,
    'platform' => 'android',
    'release_notes' => "• Fixed notification preferences\n• Improved biometric auth\n• Bug fixes",
    'download_url' => 'https://play.google.com/store/apps/details?id=com.LA DERMA.mobile',
    'min_supported_version' => '1.0.0',
    'is_active' => true,
    'is_force_update' => false,
]);

\App\Models\AppVersion::create([
    'version' => '1.0.15',
    'version_code' => 15,
    'platform' => 'ios',
    'release_notes' => "• Fixed notification preferences\n• Improved biometric auth\n• Bug fixes",
    'download_url' => 'https://apps.apple.com/app/LA DERMA/id123456789',
    'min_supported_version' => '1.0.0',
    'is_active' => true,
    'is_force_update' => false,
]);
```

### 2. Flutter App Fixes

#### Issue: Store URLs not opening correctly
The app was using web URLs that don't always work well on mobile devices.

#### Solution: Implemented proper store URL handling
- **Android**: Uses `market://` URL (opens Play Store app directly)
- **iOS**: Uses `itms-apps://` URL (opens App Store app directly)
- **Fallback**: If store app URLs fail, falls back to web URLs

Files updated:
- `lib/config/app_update_config.dart` - Added market:// and itms-apps:// URLs
- `lib/services/app_update_service.dart` - Implemented fallback mechanism

## 📱 Testing the Update System

### Step 1: Verify Database
```bash
# Check if versions are in database
php artisan tinker

# Run this:
\App\Models\AppVersion::all();
```

You should see both Android and iOS versions.

### Step 2: Test API Endpoint
```bash
# Test Android version
curl -X GET "https://LA DERMA.innovation-pulsehub.com/api/mobile/app-version" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -H "X-Platform: android" \
  -H "X-App-Version: 1.0.0"

# Test iOS version
curl -X GET "https://LA DERMA.innovation-pulsehub.com/api/mobile/app-version" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -H "X-Platform: ios" \
  -H "X-App-Version: 1.0.0"
```

Expected response:
```json
{
  "success": true,
  "data": {
    "version": "1.0.15",
    "version_code": 15,
    "platform": "android",
    "is_update_required": true,
    "is_force_update": false,
    "release_notes": "• Fixed notification preferences...",
    "download_url": "https://play.google.com/store/apps/details?id=com.LA DERMA.mobile",
    "min_supported_version": "1.0.0",
    "release_date": "2025-10-12T..."
  }
}
```

### Step 3: Test in Flutter App

#### Enable Test Mode
Edit `lib/config/app_update_config.dart`:
```dart
static const bool forceUpdateDialog = true; // Change to true for testing
```

This will always show the update dialog regardless of version.

#### Run the app and observe logs:
```
📱 AppUpdateService: Checking for updates...
📱 AppUpdateService: Current version: 1.0.14 (14)
📱 AppUpdateService: Latest version: 1.0.15
📱 AppUpdateService: Update required: true
📱 AppUpdateService: Trying Android Play Store app URL: market://details?id=com.LA DERMA.mobile
✅ AppUpdateService: Launched Play Store app successfully
```

### Step 4: Test Store URL Opening

When you click "Update Now":
1. **If Play Store/App Store is installed**: Opens directly in the store app
2. **If store app fails**: Falls back to web browser
3. **If app not yet published**: Will show "App not found" (expected)

## 🚀 Production Setup

### When Publishing to Play Store

1. **Update version in Flutter app** (`pubspec.yaml`):
```yaml
version: 1.0.15+15  # format: version+build_number
```

2. **Update database with new version**:
```bash
php artisan tinker

\App\Models\AppVersion::create([
    'version' => '1.0.15',
    'version_code' => 15,
    'platform' => 'android',
    'release_notes' => 'Your release notes here',
    'download_url' => 'https://play.google.com/store/apps/details?id=com.LA DERMA.mobile',
    'min_supported_version' => '1.0.0',
    'is_active' => true,
    'is_force_update' => false, // Set true if old versions should be forced to update
]);
```

3. **Build and upload** to Play Store:
```bash
cd /Users/esramuhannad/development/LA DERMA_mobile
flutter build appbundle --release
```

### When Publishing to App Store

1. **Get your App Store ID** from App Store Connect (looks like: `1234567890`)

2. **Update Flutter config** (`lib/config/app_update_config.dart`):
```dart
static const String iosAppId = '1234567890'; // Your actual App Store ID
```

3. **Update database**:
```bash
\App\Models\AppVersion::create([
    'version' => '1.0.15',
    'version_code' => 15,
    'platform' => 'ios',
    'release_notes' => 'Your release notes here',
    'download_url' => 'https://apps.apple.com/app/LA DERMA/id1234567890',
    'min_supported_version' => '1.0.0',
    'is_active' => true,
    'is_force_update' => false,
]);
```

## 🔧 How to Force Update

To force users to update (block old versions):

1. Set `is_force_update` to `true` in database:
```sql
UPDATE app_versions 
SET is_force_update = 1 
WHERE platform = 'android' AND version = '1.0.15';
```

2. Or via Tinker:
```php
$version = \App\Models\AppVersion::where('platform', 'android')
    ->where('version', '1.0.15')
    ->first();
$version->is_force_update = true;
$version->save();
```

When `is_force_update` is true:
- Users cannot dismiss the update dialog
- They must update to continue using the app
- Only "Update Now" button is shown (no "Skip" option)

## 📊 Version Management Best Practices

1. **Version Numbering**: Use semantic versioning (MAJOR.MINOR.PATCH)
   - `1.0.15` = Major 1, Minor 0, Patch 15

2. **Version Code**: Must increment with each release
   - Version `1.0.15` = Version code `15`
   - Version `1.1.0` = Version code `110`

3. **Minimum Supported Version**: Oldest version that can still use the app
   - Example: If min is `1.0.10`, users on `1.0.9` or below will be forced to update

4. **Release Notes**: Keep them concise and user-friendly
   - Use bullet points
   - Focus on user-facing changes
   - Avoid technical jargon

## 🐛 Troubleshooting

### Problem: 404 in Postman but popup shows in app
**Cause**: App might be using cached data or test mode
**Fix**: Check `forceUpdateDialog` setting in app config

### Problem: "Cannot connect" when clicking Update
**Cause**: App not yet published to store
**Fix**: This is normal before publishing. Once published, it will work.

### Problem: Update dialog not showing
**Causes**:
1. App version in `pubspec.yaml` is >= database version
2. `enableUpdateCheck` is set to `false`
3. API request is failing

**Fix**: Check logs and verify API is returning correct data

### Problem: Store opens but shows "App not found"
**Cause**: App not yet published or wrong package ID
**Fix**: Verify package name/bundle ID matches what's published

## 📝 API Endpoints Summary

### Get Latest Version
```
GET /api/mobile/app-version
Headers:
  - X-Platform: android|ios
  - X-App-Version: 1.0.14
```

### Create/Update Version (Admin)
```
POST /api/mobile/app-version/create
Body: {
  "version": "1.0.15",
  "version_code": 15,
  "platform": "android",
  "release_notes": "...",
  "download_url": "...",
  "min_supported_version": "1.0.0",
  "is_active": true,
  "is_force_update": false
}
```

## ✅ Checklist Before Going Live

- [ ] Database has version records for both platforms
- [ ] API endpoint returns correct data (test with curl)
- [ ] Flutter app config has correct App Store ID (iOS)
- [ ] Flutter app config has correct package name (Android)
- [ ] `forceUpdateDialog` is set to `false` in production
- [ ] Version in `pubspec.yaml` matches current release
- [ ] Apps are published to stores
- [ ] Tested update flow on both platforms
- [ ] Release notes are accurate and user-friendly

## 🎯 Quick Commands Reference

```bash
# Seed database
php artisan db:seed --class=AppVersionSeeder

# Check versions
php artisan tinker
\App\Models\AppVersion::all();

# Test API
curl -X GET "https://LA DERMA.innovation-pulsehub.com/api/mobile/app-version" -H "X-Platform: android" -H "X-App-Version: 1.0.0"

# Build Android
flutter build appbundle --release

# Build iOS
flutter build ipa --release
```

---

**Need Help?** Check the logs in the app for detailed error messages when testing the update flow.

